<?php

namespace App\Console\Commands;

use App\Services\PenaltyService;
use Illuminate\Console\Command;

class CheckPenalties extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'penalties:check 
                            {--escalate : Also check and escalate yellow cards to red}
                            {--stats : Show penalty statistics after checking}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check all active penalty rules and issue cards where needed';

    protected $penaltyService;

    /**
     * Create a new command instance.
     */
    public function __construct(PenaltyService $penaltyService)
    {
        parent::__construct();
        $this->penaltyService = $penaltyService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔍 Checking penalty rules...');
        $this->newLine();

        // Check and issue new penalties
        $issued = $this->penaltyService->checkAndIssuePenalties();

        if (count($issued) > 0) {
            $count = count($issued);
            $this->warn("⚠️  Issued {$count} penalty card(s):");
            
            foreach ($issued as $item) {
                $user = $item['user'];
                $card = $item['card'];
                $rule = $item['rule'];
                
                $icon = $card->card_type === 'yellow' ? '⚠️' : '🚫';
                $this->line("  {$icon} {$user->name} ({$user->email})");
                $this->line("     Reason: {$card->reason}");
                $this->line("     Points deducted: {$card->points_deducted}");
                $this->newLine();
            }
        } else {
            $this->info('✅ No penalties needed to be issued.');
        }

        // Check for escalations if flag is set
        if ($this->option('escalate')) {
            $this->newLine();
            $this->info('🔄 Checking for escalations...');
            
            $escalated = $this->penaltyService->checkAndEscalateCards();
            
            if (count($escalated) > 0) {
                $count = count($escalated);
                $this->error("🚫 Escalated {$count} yellow card(s) to red:");
                
                foreach ($escalated as $item) {
                    $user = $item['user'];
                    $redCard = $item['red_card'];
                    
                    $this->line("  🚫 {$user->name} ({$user->email})");
                    $this->line("     Escalated from yellow to red");
                    $this->line("     Additional points deducted: {$redCard->points_deducted}");
                    $this->newLine();
                }
            } else {
                $this->info('✅ No escalations needed.');
            }
        }

        // Show statistics if flag is set
        if ($this->option('stats')) {
            $this->newLine();
            $this->showStatistics();
        }

        $this->newLine();
        $this->info('✅ Penalty check completed!');

        return Command::SUCCESS;
    }

    /**
     * Display penalty statistics
     */
    protected function showStatistics()
    {
        $stats = $this->penaltyService->getSystemStats();

        $this->info('📊 Penalty System Statistics:');
        $this->table(
            ['Metric', 'Value'],
            [
                ['Total Cards Issued', $stats['total_cards_issued']],
                ['Active Cards', $stats['active_cards']],
                ['Yellow Cards', $stats['yellow_cards']],
                ['Red Cards', $stats['red_cards']],
                ['Total Points Deducted', $stats['total_points_deducted']],
                ['Users with Cards', $stats['users_with_cards']],
                ['Cards Issued Today', $stats['cards_today']],
                ['Escalations', $stats['escalations_count']],
            ]
        );
    }
}
