<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\CustomPage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class CustomPageController extends Controller
{
    /**
     * Display a listing of all pages.
     */
    public function index()
    {
        try {
            $pages = CustomPage::with(['creator:id,name', 'updater:id,name'])
                ->orderBy('order', 'asc')
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function ($page) {
                    return [
                        'id' => $page->id,
                        'title' => $page->title,
                        'slug' => $page->slug,
                        'is_published' => $page->is_published,
                        'is_static' => $page->is_static,
                        'template' => $page->template,
                        'order' => $page->order,
                        'published_at' => $page->published_at?->format('Y-m-d H:i:s'),
                        'created_by' => $page->creator?->name ?? 'Unknown',
                        'updated_by' => $page->updater?->name ?? 'Unknown',
                        'created_at' => $page->created_at->format('Y-m-d H:i:s'),
                        'updated_at' => $page->updated_at->format('Y-m-d H:i:s'),
                        'meta_description' => $page->meta_description,
                        'summary' => $page->getSummary(150),
                    ];
                });

            return response()->json([
                'success' => true,
                'pages' => $pages
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load pages: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created page.
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'title' => 'required|string|max:255',
                'slug' => 'nullable|string|max:255|unique:custom_pages,slug',
                'content' => 'required|array',
                'meta_description' => 'nullable|string|max:500',
                'meta_keywords' => 'nullable|string|max:255',
                'template' => 'nullable|string|max:50',
                'is_published' => 'boolean',
                'order' => 'nullable|integer',
            ]);

            // Generate slug if not provided
            if (empty($validated['slug'])) {
                $validated['slug'] = Str::slug($validated['title']);
            }

            // Set created_by
            $validated['created_by'] = Auth::id();
            $validated['updated_by'] = Auth::id();

            // Set published_at if publishing
            if ($validated['is_published'] ?? false) {
                $validated['published_at'] = now();
            }

            $page = CustomPage::create($validated);

            return response()->json([
                'success' => true,
                'message' => 'Page created successfully',
                'page' => $page
            ], 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create page: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified page.
     */
    public function show($id)
    {
        try {
            $page = CustomPage::with(['creator:id,name', 'updater:id,name'])
                ->findOrFail($id);

            return response()->json([
                'success' => true,
                'page' => [
                    'id' => $page->id,
                    'title' => $page->title,
                    'slug' => $page->slug,
                    'content' => $page->content,
                    'meta_description' => $page->meta_description,
                    'meta_keywords' => $page->meta_keywords,
                    'template' => $page->template,
                    'is_published' => $page->is_published,
                    'is_static' => $page->is_static,
                    'order' => $page->order,
                    'published_at' => $page->published_at?->format('Y-m-d H:i:s'),
                    'created_by' => $page->creator?->name ?? 'Unknown',
                    'updated_by' => $page->updater?->name ?? 'Unknown',
                    'created_at' => $page->created_at->format('Y-m-d H:i:s'),
                    'updated_at' => $page->updated_at->format('Y-m-d H:i:s'),
                ]
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Page not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load page: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified page.
     */
    public function update(Request $request, $id)
    {
        try {
            $page = CustomPage::findOrFail($id);

            // Prevent editing static pages
            if ($page->is_static) {
                return response()->json([
                    'success' => false,
                    'message' => 'Static pages cannot be edited'
                ], 403);
            }

            $validated = $request->validate([
                'title' => 'sometimes|required|string|max:255',
                'slug' => 'sometimes|required|string|max:255|unique:custom_pages,slug,' . $id,
                'content' => 'sometimes|required|array',
                'meta_description' => 'nullable|string|max:500',
                'meta_keywords' => 'nullable|string|max:255',
                'template' => 'nullable|string|max:50',
                'is_published' => 'boolean',
                'order' => 'nullable|integer',
            ]);

            // Update updated_by
            $validated['updated_by'] = Auth::id();

            // Handle publishing state change
            if (isset($validated['is_published'])) {
                if ($validated['is_published'] && !$page->is_published) {
                    // Publishing for the first time
                    $validated['published_at'] = now();
                } elseif (!$validated['is_published']) {
                    // Unpublishing
                    $validated['published_at'] = null;
                }
            }

            $page->update($validated);

            return response()->json([
                'success' => true,
                'message' => 'Page updated successfully',
                'page' => $page
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Page not found'
            ], 404);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update page: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified page.
     */
    public function destroy($id)
    {
        try {
            $page = CustomPage::findOrFail($id);

            // Prevent deleting static pages
            if ($page->is_static) {
                return response()->json([
                    'success' => false,
                    'message' => 'Static pages cannot be deleted'
                ], 403);
            }

            $title = $page->title;
            $page->delete();

            return response()->json([
                'success' => true,
                'message' => "Page '{$title}' deleted successfully"
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Page not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete page: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle page published status.
     */
    public function togglePublish($id)
    {
        try {
            $page = CustomPage::findOrFail($id);

            if ($page->is_published) {
                $page->unpublish();
                $message = 'Page unpublished successfully';
            } else {
                $page->publish();
                $message = 'Page published successfully';
            }

            return response()->json([
                'success' => true,
                'message' => $message,
                'is_published' => $page->is_published
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Page not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to toggle publish status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Duplicate an existing page.
     */
    public function duplicate($id)
    {
        try {
            $originalPage = CustomPage::findOrFail($id);

            // Cannot duplicate static pages
            if ($originalPage->is_static) {
                return response()->json([
                    'success' => false,
                    'message' => 'Static pages cannot be duplicated'
                ], 403);
            }

            $newPage = $originalPage->replicate();
            $newPage->title = $originalPage->title . ' (Copy)';
            $newPage->slug = $originalPage->slug . '-copy-' . time();
            $newPage->is_published = false;
            $newPage->published_at = null;
            $newPage->created_by = Auth::id();
            $newPage->updated_by = Auth::id();
            $newPage->save();

            return response()->json([
                'success' => true,
                'message' => 'Page duplicated successfully',
                'page' => $newPage
            ], 201);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Page not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to duplicate page: ' . $e->getMessage()
            ], 500);
        }
    }
}
