<?php

namespace App\Http\Controllers;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\AccessCode;
use App\Models\Course;
use App\Models\Lesson;
use App\Models\Topic;
use App\Models\Category;
use App\Models\SiteSetting;
use App\Services\ProgressService;
use App\Services\CourseService;
use App\Services\TopicSplitterService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class AdminController extends Controller
{
    protected $progressService;
    protected $courseService;
    protected $topicSplitter;

    public function __construct(ProgressService $progressService, CourseService $courseService, TopicSplitterService $topicSplitter)
    {
        $this->progressService = $progressService;
        $this->courseService = $courseService;
        $this->topicSplitter = $topicSplitter;
    }

    /**
     * Display the admin dashboard
     */
    public function dashboard()
    {
        // Admin access is already checked by middleware
        return view('admin.dashboard');
    }

    /**
     * Get dashboard statistics data
     */
    public function getDashboardData()
    {
        // Admin access is already checked by middleware
        $analytics = $this->progressService->getDashboardAnalytics();

        return response()->json([
            'success' => true,
            'counts' => $analytics,
            'top_performers' => $this->progressService->getTopPerformers(5),
            'at_risk_users' => $this->progressService->getAtRiskUsers(7),
        ]);
    }

    /**
     * Get all users for management with progress statistics
     */
    public function getUsers(Request $request)
    {
        // Admin access is already checked by middleware
        $users = $this->progressService->getAllUsersWithProgress();

        // Apply filters if provided
        if ($request->has('filter')) {
            $filter = $request->filter;
            $users = $users->filter(function($user) use ($filter) {
                switch($filter) {
                    case 'admin':
                        return $user['is_admin'];
                    case 'completed':
                        return $user['progress']['progress_percentage'] >= 100;
                    case 'in_progress':
                        return $user['progress']['progress_percentage'] > 0 && $user['progress']['progress_percentage'] < 100;
                    case 'not_started':
                        return $user['progress']['progress_percentage'] == 0;
                    default:
                        return true;
                }
            })->values();
        }

        return response()->json([
            'success' => true,
            'users' => $users,
            'total' => $users->count()
        ]);
    }

    /**
     * Get detailed user statistics
     */
    public function getUserDetails($userId)
    {
        $user = User::findOrFail($userId);
        $statistics = $this->progressService->getUserStatistics($user);

        return response()->json([
            'success' => true,
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'roles' => $user->roles->pluck('name')->toArray(),
                'created_at' => $user->created_at->format('Y-m-d H:i:s'),
            ],
            'statistics' => $statistics
        ]);
    }

    /**
     * Promote user to admin
     */
    public function makeAdmin(Request $request)
    {
        // Admin access is already checked by middleware
        $request->validate([
            'user_id' => 'required|exists:users,id'
        ]);

        $user = User::findOrFail($request->user_id);
        
        if (!$user->roles->contains('name', 'admin')) {
            $user->assignRole('admin');
            $user->is_admin = true;
            $user->save();
        }

        return response()->json([
            'success' => true,
            'message' => "User {$user->name} has been promoted to administrator."
        ]);
    }

    /**
     * Remove admin role from user
     */
    public function removeAdmin(Request $request)
    {
        // Admin access is already checked by middleware
        $request->validate([
            'user_id' => 'required|exists:users,id'
        ]);

        $user = User::findOrFail($request->user_id);

        // Don't allow removing admin from self
        if ($user->id === Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot remove admin privileges from yourself.'
            ], 400);
        }

        if ($user->roles->contains('name', 'admin')) {
            $user->removeRole('admin');
            $user->is_admin = false;
            $user->save();
        }

        return response()->json([
            'success' => true,
            'message' => "Admin privileges removed from {$user->name}."
        ]);
    }

    /**
     * Get all courses for management
     */
    public function getCourses(Request $request)
    {
        try {
            $languageId = $request->input('language_id');
            $courses = $this->courseService->getAllCourses($languageId);

            // Add stats for each course
            $coursesWithStats = $courses->map(function ($course) {
                $stats = $this->courseService->getCourseStats($course['id']);
                return array_merge($course, [
                    'stats' => $stats
                ]);
            });

            return response()->json([
                'success' => true,
                'courses' => $coursesWithStats
            ]);
        } catch (\Exception $e) {
            Log::error('Error in getCourses: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error fetching courses: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get a single course
     */
    public function getCourse($courseId)
    {
        try {
            $course = $this->courseService->getCourseById($courseId);

            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'course' => $course
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching course: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Create a new course
     */
    public function storeCourse(Request $request)
    {
        $chunkSettings = SiteSetting::getChunkWordsSettings();
        
        // Get active category names for validation
        $activeCategories = Category::where('is_active', true)->pluck('name')->toArray();
        $categoryRule = !empty($activeCategories) 
            ? 'nullable|string|in:' . implode(',', $activeCategories)
            : 'nullable|string|max:100';
        
        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255',
            'short_description' => 'nullable|string',
            'category' => $categoryRule,
            'chunk_words' => "nullable|integer|min:{$chunkSettings['min']}|max:{$chunkSettings['max']}",
            'certificate_template' => 'nullable|exists:certificate_templates,id',
            'completion_instructions' => 'nullable|string',
            'has_prerequisite' => 'nullable|boolean',
            'prerequisite_course_id' => 'nullable|exists:courses,id',
            'is_active' => 'nullable|boolean',
            'language_id' => 'nullable|exists:languages,id',
            'language' => 'nullable|string|max:10',
            'enable_study_time_limit' => 'nullable|boolean',
            'min_study_time_per_day' => 'nullable|integer|min:0|max:1440',
            'max_study_time_per_day' => 'nullable|integer|min:0|max:1440',
            'enable_course_survey' => 'nullable|boolean',
            'enable_survey' => 'nullable|boolean',
            'survey_mode' => 'nullable|in:feedback,survey',
            'survey_title' => 'nullable|string|max:255',
            'survey_description' => 'nullable|string',
            'is_free' => 'nullable|boolean',
            'price' => 'nullable|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            'discount_type' => 'nullable|in:amount,percent',
            'currency' => 'nullable|string|size:3',
            'image' => 'nullable|image|max:5120',
            'thumbnail' => 'nullable|image|max:5120',
        ]);

        try {
            $data = $request->all();
            $data['currency'] = isset($data['currency']) ? strtoupper(substr($data['currency'], 0, 3)) : 'USD';
            $isFree = array_key_exists('is_free', $data) ? filter_var($data['is_free'], FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) : null;
            $data['is_free'] = $isFree === null ? true : $isFree;
            if ($data['is_free']) {
                $data['price'] = 0;
                $data['discount'] = null;
                $data['discount_type'] = $data['discount_type'] ?? 'amount';
            } else {
                $data['price'] = isset($data['price']) ? (float) $data['price'] : 0;
                $data['is_free'] = false;
                $data['discount'] = isset($data['discount']) ? (float) $data['discount'] : null;
                $data['discount_type'] = $data['discount_type'] ?? 'amount';
            }

            if ($request->hasFile('image')) {
                $path = $request->file('image')->store('course-images', 'public');
                $data['image_url'] = Storage::url($path);
            }

            if ($request->hasFile('thumbnail')) {
                $path = $request->file('thumbnail')->store('course-thumbnails', 'public');
                $data['thumbnail_url'] = Storage::url($path);
            }

            $course = $this->courseService->createCourse($data);

            return response()->json([
                'success' => true,
                'message' => 'Course created successfully',
                'course' => $course
            ]);
        } catch (\Exception $e) {
            Log::error('Admin storeCourse failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request' => $request->all(),
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error creating course: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update a course
     */
    public function updateCourse(Request $request, $courseId)
    {
        $chunkSettings = SiteSetting::getChunkWordsSettings();
        
        // Get active category names for validation
        $activeCategories = Category::where('is_active', true)->pluck('name')->toArray();
        $categoryRule = !empty($activeCategories) 
            ? 'nullable|string|in:' . implode(',', $activeCategories)
            : 'nullable|string|max:100';
        
        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255',
            'short_description' => 'nullable|string',
            'category' => $categoryRule,
            'chunk_words' => "nullable|integer|min:{$chunkSettings['min']}|max:{$chunkSettings['max']}",
            'certificate_template' => 'nullable|exists:certificate_templates,id',
            'completion_instructions' => 'nullable|string',
            'has_prerequisite' => 'nullable|boolean',
            'prerequisite_course_id' => 'nullable|exists:courses,id',
            'is_active' => 'nullable|boolean',
            'language_id' => 'nullable|exists:languages,id',
            'language' => 'nullable|string|max:10',
            'enable_study_time_limit' => 'nullable|boolean',
            'min_study_time_per_day' => 'nullable|integer|min:0|max:1440',
            'max_study_time_per_day' => 'nullable|integer|min:0|max:1440',
            'enable_course_survey' => 'nullable|boolean',
            'enable_survey' => 'nullable|boolean',
            'survey_mode' => 'nullable|in:feedback,survey',
            'survey_title' => 'nullable|string|max:255',
            'survey_description' => 'nullable|string',
            'is_free' => 'nullable|boolean',
            'price' => 'nullable|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            'discount_type' => 'nullable|in:amount,percent',
            'currency' => 'nullable|string|size:3',
            'image' => 'nullable|image|max:5120',
            'thumbnail' => 'nullable|image|max:5120',
        ]);

        try {
            $existingCourse = Course::find($courseId);
            $data = $request->all();
            $data['currency'] = isset($data['currency'])
                ? strtoupper(substr($data['currency'], 0, 3))
                : (optional($existingCourse)->currency ?? 'USD');

            if (array_key_exists('is_free', $data)) {
                $isFree = filter_var($data['is_free'], FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
                $data['is_free'] = $isFree === null ? true : $isFree;
            } else {
                $data['is_free'] = optional($existingCourse)->is_free ?? true;
            }

            if ($data['is_free']) {
                $data['price'] = 0;
                $data['discount'] = null;
                $data['discount_type'] = $data['discount_type'] ?? 'amount';
            } else {
                $data['price'] = isset($data['price']) ? (float) $data['price'] : (optional($existingCourse)->price ?? 0);
                $data['is_free'] = false;
                $data['discount'] = isset($data['discount']) ? (float) $data['discount'] : (optional($existingCourse)->discount ?? null);
                $data['discount_type'] = $data['discount_type'] ?? (optional($existingCourse)->discount_type ?? 'amount');
            }

            if ($request->hasFile('image')) {
                $this->deleteStoredFile(optional($existingCourse)->image_url);
                $path = $request->file('image')->store('course-images', 'public');
                $data['image_url'] = Storage::url($path);
            }

            if ($request->hasFile('thumbnail')) {
                $this->deleteStoredFile(optional($existingCourse)->thumbnail_url);
                $path = $request->file('thumbnail')->store('course-thumbnails', 'public');
                $data['thumbnail_url'] = Storage::url($path);
            }

            $course = $this->courseService->updateCourse($courseId, $data);

            return response()->json([
                'success' => true,
                'message' => 'Course updated successfully',
                'course' => $course
            ]);
        } catch (\Exception $e) {
            Log::error('Admin updateCourse failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'courseId' => $courseId,
                'request' => $request->all(),
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error updating course: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete a course
     */
    public function destroyCourse($courseId)
    {
        try {
            $this->courseService->deleteCourse($courseId);

            return response()->json([
                'success' => true,
                'message' => 'Course and all related content deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting course: ' . $e->getMessage()
            ], 500);
        }
    }

    protected function deleteStoredFile(?string $url): void
    {
        if (!$url) {
            return;
        }

        $prefix = '/storage/';
        if (str_starts_with($url, $prefix)) {
            $relativePath = substr($url, strlen($prefix));
            Storage::disk('public')->delete($relativePath);
        }
    }

    /**
     * Get available courses for prerequisites
     */
    public function getAvailablePrerequisites($excludeCourseId = null)
    {
        try {
            $courses = $this->courseService->getAvailablePrerequisites($excludeCourseId);

            return response()->json([
                'success' => true,
                'courses' => $courses
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching available prerequisites: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get prerequisites for a course
     */
    public function getCoursePrerequisites($courseId)
    {
        try {
            $prerequisites = $this->courseService->getPrerequisites($courseId);

            return response()->json([
                'success' => true,
                'prerequisites' => $prerequisites
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching prerequisites: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get course statistics
     */
    public function getCourseStats($courseId)
    {
        try {
            $stats = $this->courseService->getCourseStats($courseId);

            if (!$stats) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'stats' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching course statistics: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get lessons for a course
     */
    public function getLessons(Course $course)
    {
        $lessons = $course->lessons()->with('topics')->orderBy('order_index')->get()->map(function ($lesson) {
            return [
                'id' => $lesson->id,
                'name' => $lesson->name,
                'intro' => $lesson->intro,
                'order_index' => $lesson->order_index,
                'share_enabled' => $lesson->share_enabled,
                'share_type' => $lesson->share_type,
                'share_url' => $lesson->share_url,
                'share_filename' => $lesson->share_filename,
                'share_size' => $lesson->share_size,
                'has_quiz' => $lesson->has_quiz ?? false,
                'has_certificate' => $lesson->has_certificate ?? false,
                'certificate_id' => $lesson->certificate_id,
                'topics_count' => $lesson->topics->count(),
                'topics' => $lesson->topics->map(function ($topic) {
                    return [
                        'id' => $topic->id,
                        'title' => $topic->title,
                        'order_index' => $topic->order_index,
                        'is_published' => $topic->is_published,
                    ];
                }),
            ];
        });

        return response()->json([
            'success' => true,
            'lessons' => $lessons
        ]);
    }

    /**
     * Get a single lesson
     */
    public function getLesson(Lesson $lesson)
    {
        return response()->json([
            'success' => true,
            'lesson' => [
                'id' => $lesson->id,
                'course_id' => $lesson->course_id,
                'name' => $lesson->name,
                'intro' => $lesson->intro,
                'order_index' => $lesson->order_index,
                'share_enabled' => $lesson->share_enabled,
                'share_type' => $lesson->share_type,
                'share_url' => $lesson->share_url,
                'share_filename' => $lesson->share_filename,
                'share_size' => $lesson->share_size,
                'has_quiz' => $lesson->has_quiz,
                'has_certificate' => $lesson->has_certificate,
                'certificate_id' => $lesson->certificate_id,
            ]
        ]);
    }

    /**
     * Create a new lesson
     */
    public function storeLesson(Request $request)
    {
        $request->validate([
            'course_id' => 'required|exists:courses,id',
            'name' => 'required|string|max:255',
            'intro' => 'nullable|string',
            'order_index' => 'nullable|integer|min:0',
            'share_enabled' => 'nullable|boolean',
            'share_type' => 'nullable|in:upload,link',
            'share_url' => 'nullable|string',
            'share_filename' => 'nullable|string',
            'share_size' => 'nullable|integer',
            'has_quiz' => 'nullable|boolean',
            'has_certificate' => 'nullable|boolean',
            'certificate_id' => 'nullable|exists:certificate_templates,id',
        ]);

        $lesson = Lesson::create([
            'course_id' => $request->course_id,
            'name' => $request->name,
            'intro' => $request->intro,
            'order_index' => $request->order_index ?? 0,
            'share_enabled' => $request->boolean('share_enabled'),
            'share_type' => $request->share_type,
            'share_url' => $request->share_url,
            'share_filename' => $request->share_filename,
            'share_size' => $request->share_size,
            'has_quiz' => $request->boolean('has_quiz'),
            'has_certificate' => $request->boolean('has_certificate'),
            'certificate_id' => $request->certificate_id,
        ]);

        // Create certificate assignment if certificate is selected
        if ($lesson->has_certificate && $lesson->certificate_id) {
            \App\Models\CertificateAssignment::updateOrCreate(
                [
                    'assignable_type' => \App\Models\Lesson::class,
                    'assignable_id' => $lesson->id,
                ],
                [
                    'certificate_template_id' => $lesson->certificate_id,
                ]
            );
        }

        return response()->json([
            'success' => true,
            'message' => 'Lesson created successfully',
            'lesson' => $lesson
        ]);
    }

    /**
     * Update a lesson
     */
    public function updateLesson(Request $request, Lesson $lesson)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'intro' => 'nullable|string',
            'order_index' => 'nullable|integer|min:0',
            'share_enabled' => 'nullable|boolean',
            'share_type' => 'nullable|in:upload,link',
            'share_url' => 'nullable|string',
            'share_filename' => 'nullable|string',
            'share_size' => 'nullable|integer',
            'has_quiz' => 'nullable|boolean',
            'has_certificate' => 'nullable|boolean',
            'certificate_id' => 'nullable|exists:certificate_templates,id',
        ]);

        $lesson->update([
            'name' => $request->name,
            'intro' => $request->intro,
            'order_index' => $request->order_index ?? 0,
            'share_enabled' => $request->boolean('share_enabled'),
            'share_type' => $request->share_type,
            'share_url' => $request->share_url,
            'share_filename' => $request->share_filename,
            'share_size' => $request->share_size,
            'has_quiz' => $request->boolean('has_quiz'),
            'has_certificate' => $request->boolean('has_certificate'),
            'certificate_id' => $request->certificate_id,
        ]);

        // Update or delete certificate assignment
        if ($lesson->has_certificate && $lesson->certificate_id) {
            \App\Models\CertificateAssignment::updateOrCreate(
                [
                    'assignable_type' => \App\Models\Lesson::class,
                    'assignable_id' => $lesson->id,
                ],
                [
                    'certificate_template_id' => $lesson->certificate_id,
                ]
            );
        } else {
            // Remove certificate assignment if unchecked
            \App\Models\CertificateAssignment::where([
                'assignable_type' => \App\Models\Lesson::class,
                'assignable_id' => $lesson->id,
            ])->delete();
        }

        return response()->json([
            'success' => true,
            'message' => 'Lesson updated successfully',
            'lesson' => $lesson
        ]);
    }

    /**
     * Delete a lesson (admin)
     */
    public function destroyLesson(Lesson $lesson)
    {
        try {
            DB::transaction(function() use ($lesson) {
                $topicIds = $lesson->topics()->pluck('id')->toArray();
                if (!empty($topicIds)) {
                    DB::table('user_progress')->whereIn('topic_id', $topicIds)->delete();
                }
                // Delete quiz questions
                $lesson->quizQuestions()->delete();
                // Delete topics and then the lesson
                $lesson->topics()->delete();
                $lesson->delete();
            });

            return response()->json([
                'success' => true,
                'message' => 'Lesson deleted successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error deleting lesson: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete lesson: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reorder lessons
     */
    public function reorderLessons(Request $request)
    {
        $request->validate([
            'lesson_ids' => 'required|array',
            'lesson_ids.*' => 'exists:lessons,id',
        ]);

        foreach ($request->lesson_ids as $index => $lessonId) {
            Lesson::where('id', $lessonId)->update(['order_index' => $index]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Lessons reordered successfully'
        ]);
    }

    /**
     * Get topics for a lesson
     */
    public function getTopics(Lesson $lesson)
    {
        $topics = $lesson->topics()->orderBy('order_index')->get()->map(function ($topic) {
            return [
                'id' => $topic->id,
                'title' => $topic->title,
                'body' => $topic->body,
                'video_url' => $topic->video_url,
                'image_url' => $topic->image_url,
                'order_index' => $topic->order_index,
                'is_published' => $topic->is_published,
            ];
        });

        return response()->json([
            'success' => true,
            'topics' => $topics
        ]);
    }

    /**
     * Get a single topic
     */
    public function getTopic(Topic $topic)
    {
        return response()->json([
            'success' => true,
            'topic' => [
                'id' => $topic->id,
                'lesson_id' => $topic->lesson_id,
                'title' => $topic->title,
                'body' => $topic->body,
                'video_url' => $topic->video_url,
                'image_url' => $topic->image_url,
                'order_index' => $topic->order_index,
                'is_published' => $topic->is_published,
            ]
        ]);
    }

    /**
     * Create a new topic
     */
    public function storeTopic(Request $request)
    {
        $request->validate([
            'lesson_id' => 'required|exists:lessons,id',
            'title' => 'required|string|max:255',
            'body' => 'nullable|string',
            'video_url' => 'nullable|string',
            'image_url' => 'nullable|string',
            'order_index' => 'nullable|integer|min:0',
            'is_published' => 'boolean',
            'auto_split' => 'nullable|boolean',
        ]);

        // Get lesson with course for chunk_words
        $lesson = Lesson::with('course')->findOrFail($request->lesson_id);
        
        // Check if auto-split is enabled (default: true)
        $autoSplit = $request->input('auto_split', true);
        $chunkWords = $lesson->course->chunk_words ?? SiteSetting::getChunkWordsDefault();
        $body = $request->input('body', '');

        // Check if splitting should occur
        if ($autoSplit && $this->topicSplitter->shouldSplit($body, $chunkWords)) {
            try {
                // Split and create multiple topics
                $topicIds = $this->topicSplitter->splitAndCreateTopics(
                    $request->lesson_id,
                    $request->title,
                    $body,
                    $chunkWords,
                    [
                        'video_url' => $request->video_url,
                        'image_url' => $request->image_url,
                        'is_published' => $request->is_published ?? true,
                        'order_index' => $request->order_index ?? 0,
                    ]
                );

                $topicCount = count($topicIds);
                $wordCount = $this->topicSplitter->getWordCount($body);

                return response()->json([
                    'success' => true,
                    'auto_split' => true,
                    'topics_created' => $topicCount,
                    'topic_ids' => $topicIds,
                    'word_count' => $wordCount,
                    'chunk_words' => $chunkWords,
                    'message' => "Content was automatically split into {$topicCount} topics based on {$chunkWords} words per topic."
                ]);
            } catch (\Exception $e) {
                \Log::error('Admin storeTopic auto-split error: ' . $e->getMessage());
                return response()->json([
                    'success' => false,
                    'message' => 'Error creating topics: ' . $e->getMessage()
                ], 500);
            }
        }

        // Create single topic (no splitting needed or disabled)
        $topic = Topic::create([
            'lesson_id' => $request->lesson_id,
            'title' => $request->title,
            'body' => $request->body,
            'video_url' => $request->video_url,
            'image_url' => $request->image_url,
            'order_index' => $request->order_index ?: 0,
            'is_published' => $request->is_published ?? true,
        ]);

        return response()->json([
            'success' => true,
            'auto_split' => false,
            'message' => 'Topic created successfully',
            'topic' => $topic,
            'id' => $topic->id,
            'topics_created' => 1
        ]);
    }

    /**
     * Update a topic
     */
    public function updateTopic(Request $request, Topic $topic)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'nullable|string',
            'video_url' => 'nullable|string',
            'image_url' => 'nullable|string',
            'order_index' => 'nullable|integer|min:0',
            'is_published' => 'boolean',
        ]);

        $topic->update([
            'title' => $request->title,
            'body' => $request->body,
            'video_url' => $request->video_url,
            'image_url' => $request->image_url,
            'order_index' => $request->order_index ?: 0,
            'is_published' => $request->is_published ?? true,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Topic updated successfully',
            'topic' => $topic
        ]);
    }

    /**
     * Delete a topic
     */
    public function destroyTopic(Topic $topic)
    {
        $topic->delete();

        return response()->json([
            'success' => true,
            'message' => 'Topic deleted successfully'
        ]);
    }

    /**
     * Split an existing topic into multiple topics
     */
    public function splitTopic($id)
    {
        $topic = Topic::with('lesson.course')->findOrFail($id);
        $chunkWords = $topic->lesson->course->chunk_words ?? SiteSetting::getChunkWordsDefault();

        // Check if splitting is possible
        if (!$this->topicSplitter->shouldSplit($topic->body, $chunkWords)) {
            $wordCount = $this->topicSplitter->getWordCount($topic->body);
            return response()->json([
                'success' => false,
                'message' => "Topic content is too short to split. Word count: {$wordCount}, Minimum required: " . ceil($chunkWords * 1.5)
            ], 400);
        }

        try {
            // Perform the split
            $topicIds = $this->topicSplitter->splitExistingTopic($topic, $chunkWords);

            return response()->json([
                'success' => true,
                'message' => 'Topic split successfully',
                'topics_created' => count($topicIds),
                'topic_ids' => $topicIds
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error splitting topic: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Preview topic split without actually creating topics
     */
    public function previewTopicSplit($id)
    {
        $topic = Topic::with('lesson.course')->findOrFail($id);
        $chunkWords = $topic->lesson->course->chunk_words ?? SiteSetting::getChunkWordsDefault();
        
        $wordCount = $this->topicSplitter->getWordCount($topic->body);
        $canSplit = $this->topicSplitter->shouldSplit($topic->body, $chunkWords);
        $topicCount = $this->topicSplitter->calculateTopicCount($topic->body, $chunkWords);

        return response()->json([
            'success' => true,
            'can_split' => $canSplit,
            'word_count' => $wordCount,
            'chunk_words' => $chunkWords,
            'topics_would_create' => $topicCount,
            'message' => $canSplit 
                ? "This topic would be split into {$topicCount} topics" 
                : "Topic content is too short to split"
        ]);
    }

    /**
     * Reorder topics
     */
    public function reorderTopics(Request $request)
    {
        $request->validate([
            'topic_ids' => 'required|array',
            'topic_ids.*' => 'exists:topics,id',
        ]);

        foreach ($request->topic_ids as $index => $topicId) {
            Topic::where('id', $topicId)->update(['order_index' => $index]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Topics reordered successfully'
        ]);
    }

    /**
     * Change user role to admin, tutor, or user
     */
    public function changeRole(Request $request)
    {
        try {
            $request->validate([
                'user_id' => 'required|exists:users,id',
                'role' => 'required|in:admin,tutor,user',
            ]);

            $user = User::findOrFail($request->user_id);

            // Prevent admin from removing their own admin role
            if ($user->id === Auth::id() && $request->role !== 'admin') {
                Log::warning('Attempt to remove own admin role', ['user_id' => $user->id, 'role' => $request->role]);
                return response()->json([
                    'success' => false,
                    'message' => 'You cannot remove your own admin privileges.'
                ], 400);
            }

            // Remove all three roles first
            $user->removeRole('admin');
            $user->removeRole('tutor');
            $user->removeRole('user');

            // Assign the selected role
            $user->assignRole($request->role);

            // Update is_admin flag
            $user->is_admin = ($request->role === 'admin');
            $user->save();

            Log::info('User role changed', ['user_id' => $user->id, 'new_role' => $request->role, 'by' => Auth::id()]);

            return response()->json([
                'success' => true,
                'message' => "User {$user->name} is now a {$request->role}."
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to change user role', [
                'error' => $e->getMessage(),
                'user_id' => $request->user_id ?? null,
                'role' => $request->role ?? null,
                'by' => Auth::id()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Failed to change user role: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get certificate templates for dropdowns
     */
    public function getCertificateTemplates()
    {
        $templates = \App\Models\CertificateTemplate::where('is_active', true)
            ->orderBy('name')
            ->get(['id', 'name', 'description']);

        return response()->json([
            'success' => true,
            'templates' => $templates
        ]);
    }
}
