<?php

namespace App\Http\Controllers;

use App\Models\PenaltyRule;
use App\Models\UserPenaltyCard;
use App\Services\PenaltyService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class AdminPenaltyController extends Controller
{
    protected $penaltyService;

    public function __construct(PenaltyService $penaltyService)
    {
        $this->penaltyService = $penaltyService;
    }

    /**
     * Get all penalty rules
     */
    public function getRules(Request $request)
    {
        try {
            $query = PenaltyRule::query();

            // Filter by active status
            if ($request->has('active')) {
                $query->where('is_active', $request->boolean('active'));
            }

            // Filter by card type
            if ($request->has('card_type')) {
                $query->where('card_type', $request->card_type);
            }

            $rules = $query->orderBy('threshold_value')->get();

            return response()->json([
                'success' => true,
                'rules' => $rules
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load penalty rules',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all issued penalty cards
     */
    public function getCards(Request $request)
    {
        try {
            $query = UserPenaltyCard::with(['user', 'penaltyRule', 'issuer'])
                ->orderBy('issued_at', 'desc');

            // Filter by card type
            if ($request->has('card_type') && $request->card_type !== 'all') {
                $query->where('card_type', $request->card_type);
            }

            // Filter by status
            if ($request->has('status') && $request->status !== 'all') {
                if ($request->status === 'active') {
                    $query->active();
                } elseif ($request->status === 'revoked') {
                    $query->whereNotNull('revoked_at');
                } elseif ($request->status === 'escalated') {
                    $query->whereNotNull('escalated_at');
                }
            }

            // Filter by user
            if ($request->has('user_id')) {
                $query->where('user_id', $request->user_id);
            }

            // Pagination
            $perPage = $request->get('per_page', 50);
            $cards = $query->paginate($perPage);

            // Transform cards for frontend
            $transformedCards = $cards->map(function ($card) {
                return [
                    'id' => $card->id,
                    'user_id' => $card->user_id,
                    'user_name' => $card->user->name ?? 'Unknown',
                    'user_email' => $card->user->email ?? '',
                    'card_type' => $card->card_type,
                    'reason' => $card->reason,
                    'points_deducted' => $card->points_deducted,
                    'issued_at' => $card->issued_at,
                    'escalated_at' => $card->escalated_at,
                    'revoked_at' => $card->revoked_at,
                    'status' => $card->revoked_at ? 'revoked' : ($card->escalated_at ? 'escalated' : 'active'),
                    'rule_name' => $card->penaltyRule->name ?? 'Unknown Rule',
                    'issued_by_name' => $card->issuer->name ?? 'System'
                ];
            });

            return response()->json([
                'success' => true,
                'cards' => $transformedCards,
                'pagination' => [
                    'total' => $cards->total(),
                    'per_page' => $cards->perPage(),
                    'current_page' => $cards->currentPage(),
                    'last_page' => $cards->lastPage()
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load penalty cards',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get penalty statistics
     */
    public function getStats(Request $request)
    {
        try {
            $stats = $this->penaltyService->getSystemStats();

            // Get recent cards for activity feed
            $recentCards = UserPenaltyCard::with(['user', 'penaltyRule'])
                ->orderBy('issued_at', 'desc')
                ->limit(10)
                ->get()
                ->map(function ($card) {
                    return [
                        'id' => $card->id,
                        'user_name' => $card->user->name ?? 'Unknown',
                        'card_type' => $card->card_type,
                        'reason' => $card->reason,
                        'points_deducted' => $card->points_deducted,
                        'issued_at' => $card->issued_at
                    ];
                });

            $stats['recent_cards'] = $recentCards;

            return response()->json([
                'success' => true,
                'stats' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load statistics',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Create a new penalty rule
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'description' => 'nullable|string',
                'card_type' => 'required|in:yellow,red',
                'trigger_type' => 'required|in:inactivity,missed_deadline,quiz_failure,attendance',
                'threshold_value' => 'required|integer|min:1',
                'points_deduction' => 'required|integer|min:0',
                'escalation_days' => 'nullable|integer|min:1',
                'escalation_points' => 'nullable|integer|min:0',
                'is_active' => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $rule = PenaltyRule::create($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Penalty rule created successfully',
                'rule' => $rule
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create penalty rule',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update a penalty rule
     */
    public function update(Request $request, $id)
    {
        try {
            $rule = PenaltyRule::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'name' => 'sometimes|string|max:255',
                'description' => 'nullable|string',
                'card_type' => 'sometimes|in:yellow,red',
                'trigger_type' => 'sometimes|in:inactivity,missed_deadline,quiz_failure,attendance',
                'threshold_value' => 'sometimes|integer|min:1',
                'points_deduction' => 'sometimes|integer|min:0',
                'escalation_days' => 'nullable|integer|min:1',
                'escalation_points' => 'nullable|integer|min:0',
                'is_active' => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $rule->update($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Penalty rule updated successfully',
                'rule' => $rule
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update penalty rule',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle a penalty rule's active status
     */
    public function toggleRule(Request $request, $id)
    {
        try {
            $rule = PenaltyRule::findOrFail($id);
            $rule->is_active = $request->boolean('is_active');
            $rule->save();

            return response()->json([
                'success' => true,
                'message' => $rule->is_active ? 'Rule activated successfully' : 'Rule deactivated successfully',
                'rule' => $rule
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to toggle penalty rule',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete a penalty rule
     */
    public function destroy($id)
    {
        try {
            $rule = PenaltyRule::findOrFail($id);
            
            // Check if rule has associated cards
            $cardsCount = $rule->penaltyCards()->count();
            if ($cardsCount > 0) {
                return response()->json([
                    'success' => false,
                    'message' => "Cannot delete rule. It has {$cardsCount} associated penalty cards. Deactivate instead."
                ], 400);
            }

            $rule->delete();

            return response()->json([
                'success' => true,
                'message' => 'Penalty rule deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete penalty rule',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Manually issue a penalty card
     */
    public function issueCard(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'user_id' => 'required|exists:users,id',
                'penalty_rule_id' => 'required|exists:penalty_rules,id',
                'reason' => 'required|string'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $rule = PenaltyRule::findOrFail($request->penalty_rule_id);
            
            $card = $this->penaltyService->issuePenaltyCard(
                $request->user_id,
                $rule,
                $request->reason,
                Auth::id()
            );

            return response()->json([
                'success' => true,
                'message' => 'Penalty card issued successfully',
                'card' => $card
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to issue penalty card',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Revoke a penalty card
     */
    public function revokeCard(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'reason' => 'required|string'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $card = UserPenaltyCard::findOrFail($id);

            if ($card->revoked_at) {
                return response()->json([
                    'success' => false,
                    'message' => 'This card has already been revoked'
                ], 400);
            }

            $this->penaltyService->revokeCard($card, $request->reason, Auth::id());

            return response()->json([
                'success' => true,
                'message' => 'Penalty card revoked successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to revoke penalty card',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Manually run penalty check
     */
    public function runCheck(Request $request)
    {
        try {
            // Run the penalty check command
            Artisan::call('penalties:check', [
                '--escalate' => true
            ]);

            $output = Artisan::output();

            // Get updated stats
            $stats = $this->penaltyService->getSystemStats();

            return response()->json([
                'success' => true,
                'message' => 'Penalty check completed successfully',
                'output' => $output,
                'stats' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to run penalty check',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get card details with history
     */
    public function getCardDetails($id)
    {
        try {
            $card = UserPenaltyCard::with(['user', 'penaltyRule', 'issuer', 'revoker', 'history.actionBy'])
                ->findOrFail($id);

            return response()->json([
                'success' => true,
                'card' => [
                    'id' => $card->id,
                    'user' => [
                        'id' => $card->user->id,
                        'name' => $card->user->name,
                        'email' => $card->user->email
                    ],
                    'rule' => [
                        'id' => $card->penaltyRule->id,
                        'name' => $card->penaltyRule->name
                    ],
                    'card_type' => $card->card_type,
                    'reason' => $card->reason,
                    'points_deducted' => $card->points_deducted,
                    'issued_at' => $card->issued_at,
                    'issued_by' => $card->issuer ? $card->issuer->name : 'System',
                    'escalated_at' => $card->escalated_at,
                    'revoked_at' => $card->revoked_at,
                    'revoked_by' => $card->revoker ? $card->revoker->name : null,
                    'revocation_reason' => $card->revocation_reason,
                    'metadata' => $card->metadata,
                    'history' => $card->history->map(function ($h) {
                        return [
                            'action_type' => $h->action_type,
                            'action_by' => $h->actionBy ? $h->actionBy->name : 'System',
                            'created_at' => $h->created_at,
                            'metadata' => $h->metadata
                        ];
                    })
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load card details',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
