<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Auth;

class PermissionController extends Controller
{
    public function index()
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json(['success' => false, 'message' => 'Admin access required'], 403);
        }

        $permissions = Permission::with('roles')->get();
        return response()->json(['success' => true, 'permissions' => $permissions]);
    }

    public function show($id)
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json(['success' => false, 'message' => 'Admin access required'], 403);
        }

        $permission = Permission::with('roles')->find($id);
        if (!$permission) return response()->json(['success' => false, 'message' => 'Permission not found'], 404);

        return response()->json(['success' => true, 'permission' => $permission]);
    }

    public function store(Request $request)
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json(['success' => false, 'message' => 'Admin access required'], 403);
        }

        $request->validate(['name' => 'required|string|max:100|unique:permissions,name']);

        $permission = Permission::create(['name' => $request->input('name')]);

        if ($request->filled('roles')) {
            $roleIds = $request->input('roles');
            $roleNames = Role::whereIn('id', $roleIds)->pluck('name')->toArray();
            $permission->syncRoles($roleNames);
        }

        return response()->json(['success' => true, 'permission' => $permission->load('roles')], 201);
    }

    public function update(Request $request, $id)
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json(['success' => false, 'message' => 'Admin access required'], 403);
        }

        $permission = Permission::find($id);
        if (!$permission) return response()->json(['success' => false, 'message' => 'Permission not found'], 404);

        $request->validate(['name' => 'required|string|max:100|unique:permissions,name,' . $id]);
        $permission->name = $request->input('name');
        $permission->save();

        if ($request->filled('roles')) {
            $roleIds = $request->input('roles');
            $roleNames = Role::whereIn('id', $roleIds)->pluck('name')->toArray();
            $permission->syncRoles($roleNames);
        }

        return response()->json(['success' => true, 'permission' => $permission->load('roles')]);
    }

    public function destroy($id)
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json(['success' => false, 'message' => 'Admin access required'], 403);
        }

        $permission = Permission::find($id);
        if (!$permission) return response()->json(['success' => false, 'message' => 'Permission not found'], 404);

        $permission->delete();
        return response()->json(['success' => true]);
    }
}
