<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\AccessCode;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class SettingsController extends Controller
{
    /**
     * Get access code requirement status
     */
    public function getAccessCodeStatus()
    {
        try {
            // Get the access_code_required setting from first row
            $setting = DB::table('settings')
                ->where('id', 1)
                ->value('access_code_required');
            
            return response()->json([
                'success' => true,
                'required' => (bool) $setting
            ]);
        } catch (\Exception $e) {
            // Fallback - assume not required if table doesn't exist
            return response()->json([
                'success' => true,
                'required' => false
            ]);
        }
    }

    /**
     * Toggle access code requirement (admin only)
     */
    public function toggleAccessCode(Request $request)
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json([
                'success' => false,
                'message' => 'Admin access required'
            ], 403);
        }

        $required = $request->boolean('required');

        try {
            // Update or insert the setting
            DB::table('settings')->updateOrInsert(
                ['id' => 1], // Assuming single row settings
                ['access_code_required' => $required ? 1 : 0]
            );

            return response()->json([
                'success' => true,
                'required' => $required
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update setting: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate access codes (admin only)
     */
    public function generateAccessCodes(Request $request)
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json([
                'success' => false,
                'message' => 'Admin access required'
            ], 403);
        }

        $request->validate([
            'count' => 'required|integer|min:1|max:100'
        ]);

        $count = $request->input('count', 1);
        $codes = [];

        try {
            for ($i = 0; $i < $count; $i++) {
                $code = strtoupper(substr(bin2hex(random_bytes(8)), 0, 12));
                
                $accessCode = AccessCode::create([
                    'code' => $code,
                    'created_by' => Auth::id(),
                    'expires_at' => now()->addHours(24), // 24 hours expiration
                    'is_used' => false
                ]);

                $codes[] = $code;
            }

            return response()->json([
                'success' => true,
                'codes' => $codes,
                'count' => count($codes)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to generate codes: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * List all access codes (admin only)
     */
    public function listAccessCodes()
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json([
                'success' => false,
                'message' => 'Admin access required'
            ], 403);
        }

        $codes = AccessCode::with(['usedBy:id,name', 'createdBy:id,name'])
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($code) {
                return [
                    'id' => $code->id,
                    'code' => $code->code,
                    'created_by' => $code->createdBy->name ?? 'Unknown',
                    'used_by' => $code->usedBy->name ?? null,
                    'expires_at' => $code->expires_at,
                    'is_used' => $code->is_used,
                    'is_expired' => $code->expires_at && $code->expires_at < now(),
                    'created_at' => $code->created_at
                ];
            });

        return response()->json([
            'success' => true,
            'codes' => $codes
        ]);
    }

    /**
     * Clear expired access codes (admin only)
     * Manual clear removes ALL expired codes immediately (no 7-day wait)
     */
    public function clearExpiredAccessCodes()
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json([
                'success' => false,
                'message' => 'Admin access required'
            ], 403);
        }

        try {
            // Delete all expired codes (no 7-day restriction for manual clearing)
            $deletedCount = AccessCode::whereNotNull('expires_at')
                ->where('expires_at', '<', now())
                ->delete();

            return response()->json([
                'success' => true,
                'message' => "Cleared {$deletedCount} expired access code(s)",
                'deleted_count' => $deletedCount
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to clear expired codes: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete individual access code (admin only)
     */
    public function deleteAccessCode($id)
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json([
                'success' => false,
                'message' => 'Admin access required'
            ], 403);
        }

        try {
            $accessCode = AccessCode::findOrFail($id);
            
            // Only allow deletion of expired codes
            if (!$accessCode->expires_at || $accessCode->expires_at >= now()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Only expired codes can be deleted individually'
                ], 400);
            }

            $code = $accessCode->code;
            $accessCode->delete();

            return response()->json([
                'success' => true,
                'message' => "Access code {$code} deleted successfully"
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Access code not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete access code: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Validate access code during registration
     */
    public function validateAccessCode(Request $request)
    {
        $code = $request->input('code');

        if (empty($code)) {
            return response()->json([
                'success' => false,
                'valid' => false,
                'message' => 'Access code is required'
            ]);
        }

        $accessCode = AccessCode::where('code', strtoupper($code))
            ->where('is_used', false)
            ->where(function ($query) {
                $query->whereNull('expires_at')
                    ->orWhere('expires_at', '>', now());
            })
            ->whereNull('used_by')
            ->first();

        if (!$accessCode) {
            return response()->json([
                'success' => false,
                'valid' => false,
                'message' => 'Invalid or expired access code'
            ]);
        }

        return response()->json([
            'success' => true,
            'valid' => true,
            'message' => 'Access code is valid'
        ]);
    }
}