<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Payment;
use App\Models\Course;
use App\Models\CourseEnrollment;
use Stripe\Stripe;
use Stripe\Checkout\Session as StripeSession;
use PayPal\Rest\ApiContext;
use PayPal\Auth\OAuthTokenCredential;
use PayPal\Api\Amount;
use PayPal\Api\Payer;
use PayPal\Api\Payment as PayPalPayment;
use PayPal\Api\RedirectUrls;
use PayPal\Api\Transaction;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class PaymentController extends Controller
{
    /**
     * Calculate the final price after discount
     */
    private function calculateFinalPrice(Course $course)
    {
        $priceValue = floatval($course->price ?? 0);
        $discountValue = floatval($course->discount ?? 0);
        $discountType = $course->discount_type ?? 'amount';
        
        if ($discountValue <= 0) {
            return $priceValue;
        }
        
        if ($discountType === 'percent') {
            $discountedPrice = $priceValue - ($priceValue * ($discountValue / 100));
        } else {
            $discountedPrice = $priceValue - $discountValue;
        }
        
        return max($discountedPrice, 0);
    }

    public function checkout(Course $course)
    {
        if ($course->is_free) {
            return redirect()->back()->with('error', 'This course is free.');
        }

        $user = Auth::user();
        
        // Check if already enrolled
        $existingEnrollment = CourseEnrollment::where('user_id', $user->id)
            ->where('course_id', $course->id)
            ->first();
            
        if ($existingEnrollment) {
            return redirect()->route('learn.course', $course->id)
                ->with('info', 'You are already enrolled in this course.');
        }

        // Get active payment gateways (using is_enabled column)
        $activeGateways = \App\Models\PaymentSetting::where('is_enabled', true)
            ->pluck('gateway')
            ->toArray();

        return view('payments.checkout', compact('course', 'activeGateways'));
    }

    public function processStripe(Request $request, Course $course)
    {
        $request->validate([
            'payment_method' => 'required|string'
        ]);

        try {
            Stripe::setApiKey(env('STRIPE_SECRET'));

            $finalPrice = $this->calculateFinalPrice($course);

            $session = StripeSession::create([
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => strtolower($course->currency),
                        'product_data' => [
                            'name' => $course->name,
                            'description' => $course->short_description,
                        ],
                        'unit_amount' => intval($finalPrice * 100), // Convert to cents
                    ],
                    'quantity' => 1,
                ]],
                'mode' => 'payment',
                'success_url' => route('payment.success', ['course' => $course->id]) . '?session_id={CHECKOUT_SESSION_ID}',
                'cancel_url' => route('payment.cancel', ['course' => $course->id]),
                'client_reference_id' => Auth::id() . '_' . $course->id,
                'metadata' => [
                    'user_id' => Auth::id(),
                    'course_id' => $course->id,
                ]
            ]);

            // Create pending payment record
            Payment::create([
                'user_id' => Auth::id(),
                'course_id' => $course->id,
                'amount' => $finalPrice,
                'currency' => $course->currency,
                'payment_method' => 'stripe',
                'transaction_id' => $session->id,
                'status' => 'pending',
                'metadata' => [
                    'session_id' => $session->id
                ]
            ]);

            return response()->json([
                'sessionId' => $session->id,
                'url' => $session->url
            ]);

        } catch (\Exception $e) {
            Log::error('Stripe payment error: ' . $e->getMessage());
            return response()->json(['error' => 'Payment processing failed. Please try again.'], 500);
        }
    }

    public function processPayPal(Request $request, Course $course)
    {
        try {
            $apiContext = new ApiContext(
                new OAuthTokenCredential(
                    env('PAYPAL_CLIENT_ID'),
                    env('PAYPAL_SECRET')
                )
            );

            $apiContext->setConfig([
                'mode' => env('PAYPAL_MODE', 'sandbox'),
            ]);

            $payer = new Payer();
            $payer->setPaymentMethod('paypal');

            $finalPrice = $this->calculateFinalPrice($course);

            $amount = new Amount();
            $amount->setCurrency($course->currency)
                ->setTotal($finalPrice);

            $transaction = new Transaction();
            $transaction->setAmount($amount)
                ->setDescription($course->name);

            $redirectUrls = new RedirectUrls();
            $redirectUrls->setReturnUrl(route('payment.paypal.success', ['course' => $course->id]))
                ->setCancelUrl(route('payment.cancel', ['course' => $course->id]));

            $payment = new PayPalPayment();
            $payment->setIntent('sale')
                ->setPayer($payer)
                ->setRedirectUrls($redirectUrls)
                ->setTransactions([$transaction]);

            $payment->create($apiContext);

            // Create pending payment record
            Payment::create([
                'user_id' => Auth::id(),
                'course_id' => $course->id,
                'amount' => $finalPrice,
                'currency' => $course->currency,
                'payment_method' => 'paypal',
                'transaction_id' => $payment->getId(),
                'status' => 'pending',
            ]);

            return response()->json([
                'approvalUrl' => $payment->getApprovalLink()
            ]);

        } catch (\Exception $e) {
            Log::error('PayPal payment error: ' . $e->getMessage());
            return response()->json(['error' => 'Payment processing failed. Please try again.'], 500);
        }
    }

    public function processPaystack(Request $request, Course $course)
    {
        try {
            $user = Auth::user();
            
            $finalPrice = $this->calculateFinalPrice($course);
            
            // Create pending payment record
            $payment = Payment::create([
                'user_id' => $user->id,
                'course_id' => $course->id,
                'amount' => $finalPrice,
                'currency' => $course->currency,
                'payment_method' => 'paystack',
                'status' => 'pending',
            ]);

            $paystackData = [
                'email' => $user->email,
                'amount' => $finalPrice * 100, // Convert to kobo
                'currency' => $course->currency,
                'reference' => 'payment_' . $payment->id . '_' . time(),
                'callback_url' => route('payment.paystack.callback'),
                'metadata' => [
                    'user_id' => $user->id,
                    'course_id' => $course->id,
                    'payment_id' => $payment->id,
                ]
            ];

            $payment->update([
                'gateway_reference' => $paystackData['reference']
            ]);

            return response()->json([
                'reference' => $paystackData['reference'],
                'public_key' => env('PAYSTACK_PUBLIC_KEY'),
                'email' => $user->email,
                'amount' => $finalPrice * 100,
                'currency' => $course->currency,
            ]);

        } catch (\Exception $e) {
            Log::error('Paystack payment error: ' . $e->getMessage());
            return response()->json(['error' => 'Payment processing failed. Please try again.'], 500);
        }
    }

    public function paystackCallback(Request $request)
    {
        $reference = $request->query('reference');
        
        if (!$reference) {
            return redirect()->route('dashboard')->with('error', 'Invalid payment reference.');
        }

        try {
            $curl = curl_init();
            curl_setopt_array($curl, [
                CURLOPT_URL => "https://api.paystack.co/transaction/verify/" . $reference,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HTTPHEADER => [
                    "Authorization: Bearer " . env('PAYSTACK_SECRET_KEY'),
                    "Cache-Control: no-cache",
                ],
            ]);

            $response = curl_exec($curl);
            curl_close($curl);

            $result = json_decode($response, true);

            if ($result['status'] && $result['data']['status'] === 'success') {
                $payment = Payment::where('gateway_reference', $reference)->first();
                
                if ($payment && $payment->isPending()) {
                    $payment->update([
                        'status' => 'completed',
                        'transaction_id' => $result['data']['id'],
                    ]);

                    // Create enrollment
                    CourseEnrollment::firstOrCreate([
                        'user_id' => $payment->user_id,
                        'course_id' => $payment->course_id,
                    ]);

                    return redirect()->route('payment.success', ['course' => $payment->course_id])
                        ->with('success', 'Payment successful! You are now enrolled.');
                }
            }

            return redirect()->route('payment.cancel', ['course' => $payment->course_id ?? null])
                ->with('error', 'Payment verification failed.');

        } catch (\Exception $e) {
            Log::error('Paystack callback error: ' . $e->getMessage());
            return redirect()->route('dashboard')->with('error', 'Payment verification failed.');
        }
    }

    public function success(Request $request, Course $course)
    {
        $sessionId = $request->query('session_id');
        
        if ($sessionId) {
            try {
                Stripe::setApiKey(env('STRIPE_SECRET'));
                $session = StripeSession::retrieve($sessionId);

                if ($session->payment_status === 'paid') {
                    $payment = Payment::where('transaction_id', $sessionId)->first();
                    
                    if ($payment && $payment->isPending()) {
                        $payment->markAsCompleted();

                        // Create enrollment
                        CourseEnrollment::firstOrCreate([
                            'user_id' => Auth::id(),
                            'course_id' => $course->id,
                        ]);
                    }
                }
            } catch (\Exception $e) {
                Log::error('Stripe success verification error: ' . $e->getMessage());
            }
        }

        return view('payments.success', compact('course'));
    }

    public function cancel(Course $course = null)
    {
        return view('payments.cancel', compact('course'));
    }

    public function paypalSuccess(Request $request, Course $course)
    {
        try {
            $paymentId = $request->query('paymentId');
            $payerId = $request->query('PayerID');

            if (!$paymentId || !$payerId) {
                return redirect()->route('payment.cancel', ['course' => $course->id])
                    ->with('error', 'Payment verification failed.');
            }

            $apiContext = new ApiContext(
                new OAuthTokenCredential(
                    env('PAYPAL_CLIENT_ID'),
                    env('PAYPAL_SECRET')
                )
            );

            $apiContext->setConfig(['mode' => env('PAYPAL_MODE', 'sandbox')]);

            $payment = PayPalPayment::get($paymentId, $apiContext);
            $execution = new \PayPal\Api\PaymentExecution();
            $execution->setPayerId($payerId);

            $result = $payment->execute($execution, $apiContext);

            if ($result->getState() === 'approved') {
                $paymentRecord = Payment::where('transaction_id', $paymentId)->first();
                
                if ($paymentRecord && $paymentRecord->isPending()) {
                    $paymentRecord->markAsCompleted();

                    // Create enrollment
                    CourseEnrollment::firstOrCreate([
                        'user_id' => Auth::id(),
                        'course_id' => $course->id,
                    ]);

                    return redirect()->route('payment.success', ['course' => $course->id])
                        ->with('success', 'Payment successful! You are now enrolled.');
                }
            }

            return redirect()->route('payment.cancel', ['course' => $course->id])
                ->with('error', 'Payment verification failed.');

        } catch (\Exception $e) {
            Log::error('PayPal success verification error: ' . $e->getMessage());
            return redirect()->route('payment.cancel', ['course' => $course->id])
                ->with('error', 'Payment processing failed.');
        }
    }

    public function history()
    {
        $payments = Payment::with('course')
            ->where('user_id', Auth::id())
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('payments.history', compact('payments'));
    }
}
