<?php

namespace App\Http\Controllers;

use App\Models\QuizQuestion;
use App\Models\Topic;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class QuizController extends Controller
{
    public function getQuiz(Request $request)
    {
        $topicId = $request->query('topic_id');

        if (!$topicId) {
            return response()->json([
                'success' => false,
                'message' => 'Missing topic_id'
            ], 400);
        }

        try {
            $topic = Topic::with('lesson')->find($topicId);

            if (!$topic) {
                return response()->json([
                    'success' => false,
                    'message' => 'Topic not found'
                ], 404);
            }

            $lessonId = $topic->lesson_id;

            $questions = QuizQuestion::where('lesson_id', $lessonId)
                ->orderBy('order_index')
                ->orderBy('id')
                ->get()
                ->map(function ($question) {
                    $options = json_decode($question->options_json, true);
                    if (!is_array($options)) {
                        $options = [];
                    }

                    return [
                        'id' => $question->id,
                        'question' => $question->question,
                        'options' => $options,
                    ];
                });

            return response()->json([
                'success' => true,
                'questions' => $questions,
                'lesson_id' => $lessonId
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading quiz: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getLessonQuiz(Request $request)
    {
        $lessonId = $request->query('lesson_id');

        if (!$lessonId) {
            return response()->json([
                'success' => false,
                'message' => 'Missing lesson_id'
            ], 400);
        }

        try {
            $questions = QuizQuestion::where('lesson_id', $lessonId)
                ->orderBy('order_index')
                ->orderBy('id')
                ->get()
                ->map(function ($question) {
                    $options = json_decode($question->options_json, true);
                    if (!is_array($options)) {
                        $options = [];
                    }

                    return [
                        'id' => $question->id,
                        'question' => $question->question,
                        'options' => $options,
                    ];
                });

            return response()->json([
                'success' => true,
                'questions' => $questions,
                'lesson_id' => $lessonId
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading lesson quiz: ' . $e->getMessage()
            ], 500);
        }
    }

    public function index(Request $request)
    {
        $lessonId = $request->query('lesson_id');
        $topicId = $request->query('topic_id');

        $query = QuizQuestion::query();

        if ($lessonId) {
            $query->where('lesson_id', $lessonId);
        }

        if ($topicId) {
            $query->where('topic_id', $topicId);
        }

        try {
            $questions = $query->orderBy('order_index')
                ->orderBy('id')
                ->get();

            return response()->json([
                'success' => true,
                'questions' => $questions
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading quiz questions: ' . $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'lesson_id' => 'nullable|exists:lessons,id',
            'topic_id' => 'nullable|exists:topics,id',
            'question' => 'required|string',
            'options_json' => 'nullable|string',
            'correct_index' => 'nullable|integer',
            'options' => 'required|string',
            'correct_answer' => 'string',
            'explanation' => 'nullable|string',
            'order_index' => 'integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        try {
            $question = QuizQuestion::create($request->all());

            return response()->json([
                'success' => true,
                'id' => $question->id
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating quiz question: ' . $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, $id)
    {
        $question = QuizQuestion::find($id);

        if (!$question) {
            return response()->json([
                'success' => false,
                'message' => 'Quiz question not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'lesson_id' => 'nullable|exists:lessons,id',
            'topic_id' => 'nullable|exists:topics,id',
            'question' => 'required|string',
            'options_json' => 'nullable|string',
            'correct_index' => 'nullable|integer',
            'options' => 'required|string',
            'correct_answer' => 'string',
            'explanation' => 'nullable|string',
            'order_index' => 'integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        try {
            $question->update($request->all());

            return response()->json([
                'success' => true,
                'id' => $question->id
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating quiz question: ' . $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        $question = QuizQuestion::find($id);

        if (!$question) {
            return response()->json([
                'success' => false,
                'message' => 'Quiz question not found'
            ], 404);
        }

        try {
            $question->delete();

            return response()->json([
                'success' => true,
                'message' => 'Quiz question deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting quiz question: ' . $e->getMessage()
            ], 500);
        }
    }
}