<?php

namespace App\Http\Controllers;

use App\Models\Course;
use App\Models\Lesson;
use App\Models\Topic;
use App\Models\UserProgress;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class StudentCourseController extends Controller
{
    /**
     * Display course details with lessons and topics
     */
    public function show($courseId)
    {
        $user = Auth::user();
        
        $course = Course::with(['lessons' => function($query) {
            $query->orderBy('order_index')->with(['topics' => function($topicQuery) {
                $topicQuery->where('is_published', true)->orderBy('order_index');
            }]);
        }])->findOrFail($courseId);

        // Check if user is enrolled
        $enrollment = \App\Models\CourseEnrollment::where('user_id', $user->id)
            ->where('course_id', $course->id)
            ->whereIn('status', ['enrolled', 'in-progress', 'completed'])
            ->first();

        // If not enrolled, return enrollment view with course info
        if (!$enrollment) {
            // Check prerequisites
            $canEnroll = true;
            $prerequisiteMessage = '';
            
            if ($course->has_prerequisite && $course->prerequisite_course_id) {
                $prerequisiteCourse = Course::find($course->prerequisite_course_id);
                $prerequisiteEnrollment = \App\Models\CourseEnrollment::where('user_id', $user->id)
                    ->where('course_id', $course->prerequisite_course_id)
                    ->where('status', 'completed')
                    ->first();
                
                if (!$prerequisiteEnrollment) {
                    $canEnroll = false;
                    $prerequisiteMessage = "You must complete '{$prerequisiteCourse->name}' before enrolling in this course.";
                }
            }

            return view('student.course-enroll', compact(
                'course',
                'canEnroll',
                'prerequisiteMessage'
            ));
        }

        // Calculate overall progress
        $totalTopics = 0;
        $completedTopics = 0;

        foreach ($course->lessons as $lesson) {
            foreach ($lesson->topics as $topic) {
                $totalTopics++;
                $progress = UserProgress::where('user_id', $user->id)
                    ->where('topic_id', $topic->id)
                    ->first();
                
                if ($progress && $progress->completed) {
                    $completedTopics++;
                }
            }
        }

        $progressPercentage = $totalTopics > 0 ? round(($completedTopics / $totalTopics) * 100) : 0;

        // Update enrollment progress
        if ($enrollment && $progressPercentage != $enrollment->progress) {
            $enrollment->updateProgress($progressPercentage);
        }

        // Auto-issue course certificate if assigned and fully completed
        if ($progressPercentage === 100) {
            try {
                \App\Http\Controllers\StudentCourseController::issueCourseCertificateIfAssigned($user->id, $course);
                
                // Mark enrollment as completed
                if ($enrollment && $enrollment->status !== 'completed') {
                    $enrollment->markAsCompleted();
                }
            } catch (\Throwable $e) {
                // Silent fail; UI remains unaffected
            }
        } elseif ($enrollment && $enrollment->status === 'enrolled' && $progressPercentage > 0) {
            // Mark as in-progress if student has started
            $enrollment->markAsInProgress();
        }

        // Get user progress for each topic
        $userProgressData = UserProgress::where('user_id', $user->id)
            ->whereIn('topic_id', $course->lessons->flatMap->topics->pluck('id'))
            ->get()
            ->keyBy('topic_id');

        // Determine which lessons are unlocked (progressive locking)
        $unlockedLessons = [];
        $allLessons = $course->lessons->sortBy('order_index')->values();
        
        foreach ($allLessons as $index => $lesson) {
            if ($index === 0) {
                // First lesson is always unlocked
                $unlockedLessons[$lesson->id] = true;
            } else {
                // Check if ALL previous lessons are completed (no skipping)
                $allPreviousCompleted = true;
                for ($i = 0; $i < $index; $i++) {
                    $previousLesson = $allLessons[$i];
                    if (!$this->isLessonCompleted($previousLesson, $user->id, $userProgressData)) {
                        $allPreviousCompleted = false;
                        break;
                    }
                }
                $unlockedLessons[$lesson->id] = $allPreviousCompleted;
            }
        }

        return view('student.course-view', compact(
            'course',
            'progressPercentage',
            'totalTopics',
            'completedTopics',
            'userProgressData',
            'unlockedLessons'
        ));
    }

    /**
     * Check if a lesson is fully completed
     */
    private function isLessonCompleted($lesson, $userId, $userProgressData)
    {
        $topics = $lesson->topics;
        
        if ($topics->count() === 0) {
            return false;
        }

        foreach ($topics as $topic) {
            $progress = $userProgressData->get($topic->id);
            if (!$progress || !$progress->completed) {
                return false;
            }
        }

        return true;
    }

    /**
     * Display a specific topic
     */
    public function viewTopic($courseId, $lessonId, $topicId)
    {
        $user = Auth::user();
        
        $course = Course::findOrFail($courseId);
        $lesson = Lesson::where('course_id', $courseId)->findOrFail($lessonId);
        $topic = Topic::where('lesson_id', $lessonId)->findOrFail($topicId);

        // Get or create user progress
        $userProgress = UserProgress::firstOrCreate(
            [
                'user_id' => $user->id,
                'topic_id' => $topic->id,
            ],
            [
                'completed' => false,
                'quiz_score' => null,
                'attempts' => 0,
            ]
        );

        // Get all topics in the lesson for navigation
        $allTopics = Topic::where('lesson_id', $lessonId)
            ->where('is_published', true)
            ->orderBy('order_index')
            ->get();

        $currentIndex = $allTopics->search(function($t) use ($topic) {
            return $t->id === $topic->id;
        });

        $previousTopic = $currentIndex > 0 ? $allTopics[$currentIndex - 1] : null;
        $nextTopic = $currentIndex < $allTopics->count() - 1 ? $allTopics[$currentIndex + 1] : null;
        $firstTopic = $allTopics->first();
        $isFirstTopicInLesson = $firstTopic && $topic->id === $firstTopic->id;

        // Check if this is the last topic in the lesson
        $isLastTopicInLesson = ($currentIndex === $allTopics->count() - 1);
        $nextLesson = null;
        $nextLessonFirstTopic = null;

        // If no next topic in this lesson, find next lesson's first topic
        if (!$nextTopic) {
            $nextLesson = Lesson::where('course_id', $courseId)
                ->where('order_index', '>', $lesson->order_index)
                ->orderBy('order_index')
                ->first();

            if ($nextLesson) {
                $nextLessonFirstTopic = Topic::where('lesson_id', $nextLesson->id)
                    ->where('is_published', true)
                    ->orderBy('order_index')
                    ->first();
                    
                $nextTopic = $nextLessonFirstTopic;
            }
        }

        // Load quiz questions if any (topic-level)
        $quizQuestions = $topic->quizQuestions()->where('is_active', true)->orderBy('order_index')->get();
        
        // Load lesson-level quiz questions if this is the last topic
        $lessonQuizQuestions = collect();
        if ($isLastTopicInLesson) {
            $lessonQuizQuestions = \App\Models\QuizQuestion::where('lesson_id', $lesson->id)
                ->whereNull('topic_id')
                ->where('is_active', true)
                ->orderBy('order_index')
                ->get();
        }

        // Calculate overall course progress for badge (published topics only)
        $totalTopics = 0;
        $completedTopics = 0;

        foreach ($course->lessons as $cLesson) {
            foreach ($cLesson->topics()->where('is_published', true)->get() as $cTopic) {
                $totalTopics++;
                $p = UserProgress::where('user_id', $user->id)
                    ->where('topic_id', $cTopic->id)
                    ->where('completed', true)
                    ->first();
                if ($p) {
                    $completedTopics++;
                }
            }
        }

        $overallProgress = $totalTopics > 0 ? round(($completedTopics / $totalTopics) * 100) : 0;

        return view('student.topic-view', compact(
            'course',
            'lesson',
            'topic',
            'userProgress',
            'previousTopic',
            'nextTopic',
            'firstTopic',
            'quizQuestions',
            'lessonQuizQuestions',
            'isLastTopicInLesson',
            'isFirstTopicInLesson',
            'nextLesson',
            'nextLessonFirstTopic',
            'overallProgress'
        ));
    }

    /**
     * Mark topic as complete
     */
    public function markComplete(Request $request, $topicId)
    {
        $user = Auth::user();
        
        $userProgress = UserProgress::updateOrCreate(
            [
                'user_id' => $user->id,
                'topic_id' => $topicId,
            ],
            [
                'completed' => true,
                'completed_at' => now(),
            ]
        );

        return response()->json([
            'success' => true,
            'message' => 'Topic marked as complete!',
            'completed' => true
        ]);
    }

    /**
     * Mark topic as incomplete
     */
    public function markIncomplete(Request $request, $topicId)
    {
        $user = Auth::user();
        
        UserProgress::where('user_id', $user->id)
            ->where('topic_id', $topicId)
            ->update([
                'completed' => false,
                'completed_at' => null,
            ]);

        return response()->json([
            'success' => true,
            'message' => 'Topic marked as incomplete',
            'completed' => false
        ]);
    }

    /**
     * Submit quiz answer
     */
    public function submitQuiz(Request $request, $topicId)
    {
        try {
            $user = Auth::user();
            $topic = Topic::findOrFail($topicId);
            
            $validated = $request->validate([
                'answers' => 'required|array',
            ]);

            $questions = $topic->quizQuestions()->where('is_active', true)->get();
            $totalQuestions = $questions->count();
            $correctAnswers = 0;

            foreach ($questions as $question) {
                $userAnswer = $validated['answers'][$question->id] ?? null;
                // Check both correct_answer (legacy) and correct_index (new format)
                $correctValue = $question->correct_index ?? $question->correct_answer ?? 0;
                
                // Convert both to integers for comparison
                if ((int)$userAnswer === (int)$correctValue) {
                    $correctAnswers++;
                }
            }

            $score = $totalQuestions > 0 ? round(($correctAnswers / $totalQuestions) * 100) : 0;
            $passed = $score >= 70; // 70% passing score

            // Update user progress
            $userProgress = UserProgress::updateOrCreate(
                [
                    'user_id' => $user->id,
                    'topic_id' => $topicId,
                ],
                [
                    'quiz_score' => $score,
                    'attempts' => DB::raw('attempts + 1'),
                    'completed' => $passed,
                    'completed_at' => $passed ? now() : null,
                ]
            );

            return response()->json([
                'success' => true,
                'score' => $score,
                'passed' => $passed,
                'correct' => $correctAnswers,
                'total' => $totalQuestions,
                'message' => $passed ? 'Congratulations! You passed!' : 'Keep trying! You need 70% to pass.'
            ]);
        } catch (\Exception $e) {
            \Log::error('Quiz submission error: ' . $e->getMessage(), [
                'topic_id' => $topicId,
                'user_id' => Auth::id(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error processing quiz: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Submit lesson quiz
     */
    public function submitLessonQuiz(Request $request, $lessonId)
    {
        try {
            $user = Auth::user();
            $lesson = Lesson::findOrFail($lessonId);
            
            $validated = $request->validate([
                'answers' => 'required|array',
            ]);

            // Get lesson-level quiz questions
            $questions = \App\Models\QuizQuestion::where('lesson_id', $lessonId)
                ->whereNull('topic_id')
                ->where('is_active', true)
                ->orderBy('order_index')
                ->get();
                
            $totalQuestions = $questions->count();
            $correctAnswers = 0;
            $results = [];

            foreach ($questions as $question) {
                $userAnswer = $validated['answers'][$question->id] ?? null;
                // Check both correct_answer (legacy) and correct_index (new format)
                $correctValue = $question->correct_index ?? $question->correct_answer ?? 0;
                
                // Convert both to integers for comparison
                $isCorrect = (int)$userAnswer === (int)$correctValue;
                
                if ($isCorrect) {
                    $correctAnswers++;
                }
                
                $results[] = [
                    'question_id' => $question->id,
                    'correct' => $isCorrect,
                    'user_answer' => $userAnswer,
                    'correct_answer' => $correctValue
                ];
            }

            $score = $totalQuestions > 0 ? round(($correctAnswers / $totalQuestions) * 100) : 0;
            $passed = $score >= 70; // 70% passing score

            // Store quiz result for last topic in lesson (for progress tracking)
            $lastTopic = $lesson->topics()->orderBy('order_index', 'desc')->first();
            
            if ($lastTopic) {
                UserProgress::updateOrCreate(
                    [
                        'user_id' => $user->id,
                        'topic_id' => $lastTopic->id,
                    ],
                    [
                        'quiz_score' => $score,
                        'attempts' => DB::raw('attempts + 1'),
                        'completed' => $passed,
                        'completed_at' => $passed ? now() : null,
                    ]
                );
            }

            return response()->json([
                'success' => true,
                'score' => $score,
                'passed' => $passed,
                'correct' => $correctAnswers,
                'total' => $totalQuestions,
                'results' => $results,
                'message' => $passed ? 'Congratulations! You passed!' : 'Keep trying! You need 70% to pass.'
            ]);
        } catch (\Exception $e) {
            \Log::error('Lesson quiz submission error: ' . $e->getMessage(), [
                'lesson_id' => $lessonId,
                'user_id' => Auth::id(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error processing lesson quiz: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Check if a lesson is fully completed
     */
    public function checkLessonCompletion($lessonId)
    {
        $user = Auth::user();
        $lesson = Lesson::with('topics')->findOrFail($lessonId);
        
        $topics = $lesson->topics;
        $allCompleted = true;

        foreach ($topics as $topic) {
            $progress = UserProgress::where('user_id', $user->id)
                ->where('topic_id', $topic->id)
                ->first();
            
            if (!$progress || !$progress->completed) {
                $allCompleted = false;
                break;
            }
        }

        // If completed, attempt auto-issuance for lesson certificate (if assigned)
        if ($allCompleted) {
            try {
                \App\Http\Controllers\StudentCourseController::issueLessonCertificateIfAssigned($user->id, $lesson);
            } catch (\Throwable $e) {
                // Fail silently; certificate issuance is best-effort here
            }
        }

        return response()->json([
            'completed' => $allCompleted,
            'lesson_id' => $lessonId
        ]);
    }

    /**
     * Attempt to issue a lesson certificate if an assignment exists and not yet issued.
     */
    public static function issueLessonCertificateIfAssigned($userId, Lesson $lesson)
    {
        $assignment = \App\Models\CertificateAssignment::where([
            'assignable_type' => \App\Models\Lesson::class,
            'assignable_id' => $lesson->id,
        ])->first();

        if (!$assignment) {
            return;
        }

        $already = \App\Models\IssuedCertificate::where([
            'user_id' => $userId,
            'assignable_type' => \App\Models\Lesson::class,
            'assignable_id' => $lesson->id,
            'certificate_template_id' => $assignment->certificate_template_id,
        ])->exists();

        if ($already) {
            return;
        }

        $user = \App\Models\User::find($userId);
        if (!$user) return;

        $generator = app(\App\Services\CertificateGeneratorService::class);
        $generator->generateCertificate($user, $assignment->template, $lesson);
    }

    /**
     * Attempt to issue a course certificate if an assignment exists and not yet issued.
     */
    public static function issueCourseCertificateIfAssigned($userId, Course $course)
    {
        $assignment = \App\Models\CertificateAssignment::where([
            'assignable_type' => \App\Models\Course::class,
            'assignable_id' => $course->id,
        ])->first();

        if (!$assignment) {
            return;
        }

        $already = \App\Models\IssuedCertificate::where([
            'user_id' => $userId,
            'assignable_type' => \App\Models\Course::class,
            'assignable_id' => $course->id,
            'certificate_template_id' => $assignment->certificate_template_id,
        ])->exists();

        if ($already) {
            return;
        }

        $user = \App\Models\User::find($userId);
        if (!$user) return;

        $generator = app(\App\Services\CertificateGeneratorService::class);
        $generator->generateCertificate($user, $assignment->template, $course);
    }
}
