<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;
use App\Models\User;

class StudentProfileController extends Controller
{
    /**
     * Show student profile page
     */
    public function index()
    {
        $user = Auth::user();
        
        // Get user's enrollment stats
        $enrolledCoursesCount = \App\Models\Course::whereHas('lessons.topics.userProgress', function($query) use ($user) {
            $query->where('user_id', $user->id);
        })->count();
        
        $completedTopics = \App\Models\UserProgress::where('user_id', $user->id)
            ->where('completed', true)
            ->count();
        
        $totalTimeSpent = \App\Models\UserProgress::where('user_id', $user->id)
            ->sum('video_watch_seconds') ?? 0;
        
        $averageScore = \App\Models\UserProgress::where('user_id', $user->id)
            ->whereNotNull('quiz_score')
            ->avg('quiz_score') ?? 0;
        
        // Get recent activity
        $recentActivity = \App\Models\UserProgress::where('user_id', $user->id)
            ->with(['topic.lesson.course'])
            ->orderBy('updated_at', 'desc')
            ->limit(10)
            ->get();
        
        // Get certificates
        $certificates = \App\Models\IssuedCertificate::where('user_id', $user->id)
            ->with('course')
            ->orderBy('issued_at', 'desc')
            ->get();
        
        return view('student.profile', compact(
            'user',
            'enrolledCoursesCount',
            'completedTopics',
            'totalTimeSpent',
            'averageScore',
            'recentActivity',
            'certificates'
        ));
    }
    
    /**
     * Update student profile
     */
    public function update(Request $request)
    {
        try {
            $user = Auth::user();
            
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|email|max:255|unique:users,email,' . $user->id,
                'bio' => 'nullable|string|max:500',
                'phone' => 'nullable|string|max:20',
                'location' => 'nullable|string|max:100',
                'language' => 'nullable|string|max:10',
                'timezone' => 'nullable|string|max:50',
            ]);
            
            $user->update($validated);
            
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Profile updated successfully!',
                    'reload' => true
                ]);
            }
            
            return back()->with('success', 'Profile updated successfully!');
        } catch (ValidationException $e) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $e->errors()
                ], 422);
            }
            throw $e;
        } catch (\Exception $e) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error updating profile: ' . $e->getMessage()
                ], 500);
            }
            return back()->withErrors(['error' => 'Error updating profile']);
        }
    }
    
    /**
     * Update student avatar
     */
    public function updateAvatar(Request $request)
    {
        try {
            $user = Auth::user();
            
            $request->validate([
                'avatar' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
            ]);
            
            // Delete old avatar if exists
            if ($user->avatar && Storage::disk('public')->exists($user->avatar)) {
                Storage::disk('public')->delete($user->avatar);
            }
            
            // Store new avatar
            $path = $request->file('avatar')->store('avatars', 'public');
            
            $user->update(['avatar' => $path]);
            
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Avatar updated successfully!',
                    'reload' => true,
                    'avatar_url' => Storage::url($path)
                ]);
            }
            
            return back()->with('success', 'Avatar updated successfully!');
        } catch (ValidationException $e) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $e->errors()
                ], 422);
            }
            throw $e;
        } catch (\Exception $e) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error uploading avatar: ' . $e->getMessage()
                ], 500);
            }
            return back()->withErrors(['avatar' => 'Error uploading avatar']);
        }
    }
    
    /**
     * Change password
     */
    public function changePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'new_password' => 'required|min:8|confirmed',
        ]);
        
        $user = Auth::user();
        
        // Check if current password is correct
        if (!Hash::check($request->current_password, $user->password)) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Current password is incorrect.'
                ], 422);
            }
            return back()->withErrors(['current_password' => 'Current password is incorrect.']);
        }
        
        // Update password
        $user->update([
            'password' => Hash::make($request->new_password)
        ]);
        
        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Password changed successfully!'
            ]);
        }
        
        return back()->with('success', 'Password changed successfully!');
    }
    
    /**
     * Show settings page
     */
    public function settings()
    {
        $user = Auth::user();
        
        // Get user preferences/settings (you can expand this)
        $settings = [
            'email_notifications' => $user->email_notifications ?? true,
            'push_notifications' => $user->push_notifications ?? true,
            'course_reminders' => $user->course_reminders ?? true,
            'achievement_alerts' => $user->achievement_alerts ?? true,
            'newsletter' => $user->newsletter ?? true,
        ];
        
        return view('student.settings', compact('user', 'settings'));
    }
    
    /**
     * Update settings
     */
    public function updateSettings(Request $request)
    {
        $user = Auth::user();
        
        $validated = $request->validate([
            'email_notifications' => 'boolean',
            'push_notifications' => 'boolean',
            'course_reminders' => 'boolean',
            'achievement_alerts' => 'boolean',
            'newsletter' => 'boolean',
        ]);
        
        $user->update($validated);
        
        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Settings updated successfully!'
            ]);
        }
        
        return back()->with('success', 'Settings updated successfully!');
    }
    
    /**
     * Update privacy settings
     */
    public function updatePrivacy(Request $request)
    {
        $user = Auth::user();
        
        $validated = $request->validate([
            'profile_visible' => 'boolean',
            'show_leaderboard' => 'boolean',
            'share_progress' => 'boolean',
        ]);
        
        $user->update($validated);
        
        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Privacy settings updated successfully!'
            ]);
        }
        
        return back()->with('success', 'Privacy settings updated successfully!');
    }
}
