<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Models\Course;
use App\Models\Lesson;
use App\Models\Topic;
use App\Models\User;
use App\Models\UserProgress;
use App\Models\UserLessonProgress;
use App\Models\QuizQuestion;
use App\Models\SiteSetting;
use App\Services\CourseService;
use App\Services\TopicSplitterService;
use Illuminate\Support\Facades\Storage;

class TutorController extends Controller
{
    protected $courseService;
    protected $topicSplitter;

    public function __construct(CourseService $courseService, TopicSplitterService $topicSplitter)
    {
        $this->courseService = $courseService;
        $this->topicSplitter = $topicSplitter;
    }

    // Get all courses for the tutor
    public function getCourses(Request $request)
    {
        $courses = Course::with(['lessons.topics', 'language'])
            ->where('created_by', Auth::id())
            ->when($request->language_id, function($query) use ($request) {
                $query->where('language_id', $request->language_id);
            })
            ->select([
                'id', 'name', 'slug', 'short_description', 'category', 'is_active', 
                'chunk_words', 'certificate_template', 'completion_instructions',
                'has_prerequisite', 'prerequisite_course_id',
                'is_free', 'price', 'discount', 'discount_type', 'currency', 'image_url', 'thumbnail_url',
                'created_by', 'created_at', 'updated_at',
                'enable_course_survey', 'survey_mode', 'survey_title', 'survey_description',
                'language_id',
            ])
            ->orderBy('created_at', 'desc')
            ->get();
        $courses->transform(function ($course) {
            $course->language_code = optional($course->language)->code;
            $course->language_name = optional($course->language)->name;
            $course->language_flag_emoji = optional($course->language)->flag_emoji;
            return $course;
        });

        return response()->json($courses);
    }

    // Get single course details
    public function getCourse($id)
    {
        $course = Course::with(['lessons.topics', 'lessons.quizQuestions', 'language'])
            ->where('created_by', Auth::id())
            ->findOrFail($id);
        $course->language_code = optional($course->language)->code;
        $course->language_name = optional($course->language)->name;
        $course->language_flag_emoji = optional($course->language)->flag_emoji;

        return response()->json($course);
    }

    // Create new course
    public function storeCourse(Request $request)
    {
        $chunkSettings = SiteSetting::getChunkWordsSettings();
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'short_description' => 'nullable|string',
            'category' => 'nullable|string|max:255',
            'chunk_words' => "nullable|integer|min:{$chunkSettings['min']}|max:{$chunkSettings['max']}",
            'certificate_template' => 'nullable|exists:certificate_templates,id',
            'completion_instructions' => 'nullable|string',
            'has_prerequisite' => 'nullable|boolean',
            'prerequisite_course_id' => 'nullable|exists:courses,id',
            'is_active' => 'boolean',
            'language_id' => 'nullable|exists:languages,id',
            'enable_study_time_limit' => 'nullable|boolean',
            'min_study_time_per_day' => 'nullable|integer|min:0|max:1440',
            'max_study_time_per_day' => 'nullable|integer|min:0|max:1440',
            'enable_course_survey' => 'nullable|boolean',
            'survey_mode' => 'nullable|in:feedback,survey',
            'survey_title' => 'nullable|string|max:255',
            'survey_description' => 'nullable|string',
            'is_free' => 'nullable|boolean',
            'price' => 'nullable|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            'discount_type' => 'nullable|in:amount,percent',
            'currency' => 'nullable|string|size:3',
            'image' => 'nullable|image|max:5120',
            'thumbnail' => 'nullable|image|max:5120',
        ]);

        $slug = Str::slug($validated['name']);
        $originalSlug = $slug;
        $counter = 1;
        while (Course::where('slug', $slug)->exists()) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        $validated['created_by'] = Auth::id();
        $validated['is_active'] = $request->boolean('is_active', true);
        $validated['slug'] = $slug;
        $validated['currency'] = strtoupper(substr($request->input('currency', 'USD'), 0, 3));
        
        // Set chunk_words default if not provided
        if (!isset($validated['chunk_words']) || empty($validated['chunk_words'])) {
            $validated['chunk_words'] = SiteSetting::getChunkWordsDefault();
        }
        
        // Handle prerequisite
        $validated['has_prerequisite'] = $request->boolean('has_prerequisite', false);
        if (!$validated['has_prerequisite']) {
            $validated['prerequisite_course_id'] = null;
        }
        
        // Handle survey settings
        $validated['enable_course_survey'] = $request->boolean('enable_survey', false);
        if (!$validated['enable_course_survey']) {
            $validated['survey_mode'] = null;
            $validated['survey_title'] = null;
            $validated['survey_description'] = null;
        } else {
            $validated['survey_mode'] = $request->input('survey_mode', 'feedback');
        }
        
        // Handle study time limit settings
        $validated['enable_study_time_limit'] = $request->boolean('enable_study_time_limit', false);
        if (!$validated['enable_study_time_limit']) {
            $validated['min_study_time_per_day'] = null;
            $validated['max_study_time_per_day'] = null;
        }

        $isFree = $request->has('is_free')
            ? filter_var($request->input('is_free'), FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE)
            : null;
        $validated['is_free'] = $isFree === null ? true : $isFree;
        if ($validated['is_free']) {
            $validated['price'] = 0;
            $validated['discount'] = null;
            $validated['discount_type'] = 'amount';
        } else {
            $validated['price'] = (float) $request->input('price', 0);
            $validated['is_free'] = false;
            $validated['discount'] = $request->filled('discount') ? (float) $request->input('discount') : null;
            $validated['discount_type'] = $request->input('discount_type', 'amount');
        }

        if ($request->hasFile('image')) {
            $validated['image_url'] = Storage::url($request->file('image')->store('course-images', 'public'));
        }

        if ($request->hasFile('thumbnail')) {
            $validated['thumbnail_url'] = Storage::url($request->file('thumbnail')->store('course-thumbnails', 'public'));
        }

        // Resolve language (accepts code or ID) with default fallback
        $languageId = $validated['language_id'] ?? null;
        if (!$languageId && $request->filled('language')) {
            $language = \App\Models\Language::where('code', $request->input('language'))->first();
            $languageId = $language ? $language->id : null;
        }
        if (!$languageId) {
            $defaultLanguage = \App\Models\Language::getDefault();
            $languageId = $defaultLanguage ? $defaultLanguage->id : null;
        }
        $validated['language_id'] = $languageId;

        $course = Course::create($validated);

        // Create certificate assignment if certificate is selected
        if (!empty($validated['certificate_template'])) {
            \App\Models\CertificateAssignment::updateOrCreate(
                [
                    'assignable_type' => \App\Models\Course::class,
                    'assignable_id' => $course->id,
                ],
                [
                    'certificate_template_id' => $validated['certificate_template'],
                ]
            );
        }

        return response()->json([
            'success' => true,
            'message' => 'Course created successfully',
            'course' => $course
        ]);
    }

    // Update course
    public function updateCourse(Request $request, $id)
    {
        $course = Course::where('created_by', Auth::id())->findOrFail($id);
        
        $chunkSettings = SiteSetting::getChunkWordsSettings();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'short_description' => 'nullable|string',
            'category' => 'nullable|string|max:255',
            'chunk_words' => "nullable|integer|min:{$chunkSettings['min']}|max:{$chunkSettings['max']}",
            'certificate_template' => 'nullable|exists:certificate_templates,id',
            'completion_instructions' => 'nullable|string',
            'has_prerequisite' => 'nullable|boolean',
            'prerequisite_course_id' => 'nullable|exists:courses,id',
            'is_active' => 'boolean',
            'language_id' => 'nullable|exists:languages,id',
            'enable_study_time_limit' => 'nullable|boolean',
            'min_study_time_per_day' => 'nullable|integer|min:0|max:1440',
            'max_study_time_per_day' => 'nullable|integer|min:0|max:1440',
            'enable_course_survey' => 'nullable|boolean',
            'survey_mode' => 'nullable|in:feedback,survey',
            'survey_title' => 'nullable|string|max:255',
            'survey_description' => 'nullable|string',
            'is_free' => 'nullable|boolean',
            'price' => 'nullable|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            'discount_type' => 'nullable|in:amount,percent',
            'currency' => 'nullable|string|size:3',
            'image' => 'nullable|image|max:5120',
            'thumbnail' => 'nullable|image|max:5120',
        ]);

        if (isset($validated['name']) && $validated['name'] !== $course->name) {
            $slug = Str::slug($validated['name']);
            $originalSlug = $slug;
            $counter = 1;
            while (Course::where('slug', $slug)->where('id', '!=', $course->id)->exists()) {
                $slug = $originalSlug . '-' . $counter;
                $counter++;
            }
            $validated['slug'] = $slug;
        }
        if (isset($validated['is_active'])) {
            $validated['is_active'] = $request->boolean('is_active');
        }

        $validated['currency'] = isset($validated['currency'])
            ? strtoupper(substr($validated['currency'], 0, 3))
            : ($course->currency ?? 'USD');
        
        // Set chunk_words default if not provided
        if (!isset($validated['chunk_words']) || empty($validated['chunk_words'])) {
            $validated['chunk_words'] = $course->chunk_words ?? SiteSetting::getChunkWordsDefault();
        }
        
        // Handle prerequisite
        if (array_key_exists('has_prerequisite', $validated)) {
            $validated['has_prerequisite'] = filter_var($validated['has_prerequisite'], FILTER_VALIDATE_BOOLEAN);
            if (!$validated['has_prerequisite']) {
                $validated['prerequisite_course_id'] = null;
            }
        }
        
        // Handle survey settings
        if (array_key_exists('enable_course_survey', $validated)) {
            $validated['enable_course_survey'] = filter_var($validated['enable_course_survey'], FILTER_VALIDATE_BOOLEAN);
            if (!$validated['enable_course_survey']) {
                $validated['survey_mode'] = null;
                $validated['survey_title'] = null;
                $validated['survey_description'] = null;
            } else {
                $validated['survey_mode'] = $request->input('survey_mode', $course->survey_mode ?? 'feedback');
            }
        }
        
        // Handle study time limit settings
        if (array_key_exists('enable_study_time_limit', $validated)) {
            $validated['enable_study_time_limit'] = filter_var($validated['enable_study_time_limit'], FILTER_VALIDATE_BOOLEAN);
            if (!$validated['enable_study_time_limit']) {
                $validated['min_study_time_per_day'] = null;
                $validated['max_study_time_per_day'] = null;
            }
        }

        if (array_key_exists('is_free', $validated)) {
            $isFree = filter_var($validated['is_free'], FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
            $validated['is_free'] = $isFree === null ? true : $isFree;
        } else {
            $validated['is_free'] = $course->is_free ?? true;
        }

        if ($validated['is_free']) {
            $validated['price'] = 0;
            $validated['discount'] = null;
            $validated['discount_type'] = 'amount';
        } else {
            $validated['price'] = isset($validated['price']) ? (float) $validated['price'] : ($course->price ?? 0);
            $validated['is_free'] = false;
            $validated['discount'] = $request->filled('discount')
                ? (float) $request->input('discount')
                : ($course->discount ?? null);
            $validated['discount_type'] = $request->input('discount_type', $course->discount_type ?? 'amount');
        }

        if ($request->hasFile('image')) {
            $this->deleteStoredFile($course->image_url);
            $validated['image_url'] = Storage::url($request->file('image')->store('course-images', 'public'));
        }

        if ($request->hasFile('thumbnail')) {
            $this->deleteStoredFile($course->thumbnail_url);
            $validated['thumbnail_url'] = Storage::url($request->file('thumbnail')->store('course-thumbnails', 'public'));
        }

        // Resolve language (accepts code or ID) with default to existing or system default
        $languageId = $validated['language_id'] ?? null;
        if (!$languageId && $request->filled('language')) {
            $language = \App\Models\Language::where('code', $request->input('language'))->first();
            $languageId = $language ? $language->id : null;
        }
        if (!$languageId) {
            $languageId = $course->language_id;
        }
        if (!$languageId) {
            $defaultLanguage = \App\Models\Language::getDefault();
            $languageId = $defaultLanguage ? $defaultLanguage->id : null;
        }
        $validated['language_id'] = $languageId;

        $course->update($validated);

        // Update or delete certificate assignment
        if (!empty($validated['certificate_template'])) {
            \App\Models\CertificateAssignment::updateOrCreate(
                [
                    'assignable_type' => \App\Models\Course::class,
                    'assignable_id' => $course->id,
                ],
                [
                    'certificate_template_id' => $validated['certificate_template'],
                ]
            );
        } else {
            // Remove certificate assignment if empty
            \App\Models\CertificateAssignment::where([
                'assignable_type' => \App\Models\Course::class,
                'assignable_id' => $course->id,
            ])->delete();
        }

        return response()->json([
            'success' => true,
            'message' => 'Course updated successfully',
            'course' => $course
        ]);
    }

    // Delete course
    public function destroyCourse($id)
    {
        $course = Course::where('created_by', Auth::id())->findOrFail($id);
        
        try {
            DB::transaction(function() use ($course) {
                // Delete all lessons and their related data
                foreach ($course->lessons as $lesson) {
                    // Delete quiz questions
                    $lesson->quizQuestions()->delete();
                    // Delete topics
                    $lesson->topics()->delete();
                    // Delete lesson
                    $lesson->delete();
                }
                
                // Delete the course
                $course->delete();
            });

            return response()->json([
                'success' => true,
                'message' => 'Course deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete course: ' . $e->getMessage()
            ], 500);
        }
    }

    // Get lessons for a course
    public function getLessons($courseId)
    {
        // Verify tutor owns this course
        $course = Course::where('created_by', Auth::id())->findOrFail($courseId);
        
        $lessons = Lesson::where('course_id', $courseId)
            ->with('topics')
            ->orderBy('order_index')
            ->get();

        return response()->json($lessons);
    }

    // Create new lesson
    public function storeLesson(Request $request)
    {
        $validated = $request->validate([
            'course_id' => 'required|exists:courses,id',
            'name' => 'required|string|max:255',
            'intro' => 'nullable|string',
            'order_index' => 'nullable|integer',
            'has_quiz' => 'nullable|boolean',
            'has_certificate' => 'nullable|boolean',
            'certificate_id' => 'nullable|exists:certificate_templates,id',
        ]);

        // Verify tutor owns this course
        $course = Course::where('created_by', Auth::id())->findOrFail($validated['course_id']);

        $lesson = Lesson::create([
            'course_id' => $validated['course_id'],
            'name' => $validated['name'],
            'intro' => $validated['intro'] ?? null,
            'order_index' => $validated['order_index'] ?? 0,
            'has_quiz' => $validated['has_quiz'] ?? false,
            'has_certificate' => $validated['has_certificate'] ?? false,
            'certificate_id' => $validated['certificate_id'] ?? null,
        ]);

        // Create certificate assignment if certificate is selected
        if ($lesson->has_certificate && $lesson->certificate_id) {
            \App\Models\CertificateAssignment::updateOrCreate(
                [
                    'assignable_type' => \App\Models\Lesson::class,
                    'assignable_id' => $lesson->id,
                ],
                [
                    'certificate_template_id' => $lesson->certificate_id,
                ]
            );
        }

        return response()->json([
            'success' => true,
            'message' => 'Lesson created successfully',
            'lesson' => $lesson
        ]);
    }

    // Update lesson
    public function updateLesson(Request $request, $id)
    {
        $lesson = Lesson::findOrFail($id);
        
        // Verify tutor owns the course this lesson belongs to
        $course = Course::where('created_by', Auth::id())->findOrFail($lesson->course_id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'intro' => 'nullable|string',
            'order_index' => 'nullable|integer',
            'has_quiz' => 'nullable|boolean',
            'has_certificate' => 'nullable|boolean',
            'certificate_id' => 'nullable|exists:certificate_templates,id',
        ]);

        $updateData = [
            'name' => $validated['name'],
            'intro' => $validated['intro'] ?? null,
            'has_quiz' => $validated['has_quiz'] ?? false,
            'has_certificate' => $validated['has_certificate'] ?? false,
            'certificate_id' => $validated['certificate_id'] ?? null,
        ];
        if (array_key_exists('order_index', $validated)) {
            $updateData['order_index'] = $validated['order_index'];
        }

        $lesson->update($updateData);

        // Update or delete certificate assignment
        if ($lesson->has_certificate && $lesson->certificate_id) {
            \App\Models\CertificateAssignment::updateOrCreate(
                [
                    'assignable_type' => \App\Models\Lesson::class,
                    'assignable_id' => $lesson->id,
                ],
                [
                    'certificate_template_id' => $lesson->certificate_id,
                ]
            );
        } else {
            // Remove certificate assignment if unchecked
            \App\Models\CertificateAssignment::where([
                'assignable_type' => \App\Models\Lesson::class,
                'assignable_id' => $lesson->id,
            ])->delete();
        }

        return response()->json([
            'success' => true,
            'message' => 'Lesson updated successfully',
            'lesson' => $lesson
        ]);
    }

    // Delete lesson
    public function destroyLesson($id)
    {
        $lesson = Lesson::findOrFail($id);
        
        // Verify tutor owns the course this lesson belongs to
        $course = Course::where('created_by', Auth::id())->findOrFail($lesson->course_id);
        
        // Delete related quiz questions first (cascade)
        $lesson->quizQuestions()->delete();
        
        // Delete related topics
        $lesson->topics()->delete();
        
        // Now delete the lesson
        $lesson->delete();

        return response()->json([
            'success' => true,
            'message' => 'Lesson deleted successfully'
        ]);
    }

    // Get topics for a lesson
    public function getTopics($lessonId)
    {
        $lesson = Lesson::findOrFail($lessonId);
        
        // Verify tutor owns the course this lesson belongs to
        $course = Course::where('created_by', Auth::id())->findOrFail($lesson->course_id);
        
        $topics = Topic::where('lesson_id', $lessonId)
            ->orderBy('order_index')
            ->get();

        return response()->json($topics);
    }

    // Create new topic
    public function storeTopic(Request $request)
    {
        $validated = $request->validate([
            'lesson_id' => 'required|exists:lessons,id',
            'title' => 'required|string|max:255',
            'body' => 'nullable|string',
            'video_url' => 'nullable|string',
            'image_url' => 'nullable|string',
            'is_published' => 'nullable|boolean',
            'order_index' => 'nullable|integer',
            'auto_split' => 'nullable|boolean',
        ]);

        $lesson = Lesson::with('course')->findOrFail($validated['lesson_id']);
        
        // Verify tutor owns the course this lesson belongs to
        $course = Course::where('created_by', Auth::id())->findOrFail($lesson->course_id);

        // Check if auto-split is enabled (default: true)
        $autoSplit = $validated['auto_split'] ?? true;
        $chunkWords = $course->chunk_words ?? SiteSetting::getChunkWordsDefault();
        $body = $validated['body'] ?? '';

        // Check if splitting should occur
        if ($autoSplit && $this->topicSplitter->shouldSplit($body, $chunkWords)) {
            try {
                // Split and create multiple topics
                $topicIds = $this->topicSplitter->splitAndCreateTopics(
                    $validated['lesson_id'],
                    $validated['title'],
                    $body,
                    $chunkWords,
                    [
                        'video_url' => $validated['video_url'] ?? null,
                        'image_url' => $validated['image_url'] ?? null,
                        'is_published' => $validated['is_published'] ?? true,
                        'order_index' => $validated['order_index'] ?? 0,
                    ]
                );

                $topicCount = count($topicIds);
                $wordCount = $this->topicSplitter->getWordCount($body);

                return response()->json([
                    'success' => true,
                    'auto_split' => true,
                    'topics_created' => $topicCount,
                    'topic_ids' => $topicIds,
                    'word_count' => $wordCount,
                    'chunk_words' => $chunkWords,
                    'message' => "Content was automatically split into {$topicCount} topics based on {$chunkWords} words per topic."
                ]);
            } catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error creating topics: ' . $e->getMessage()
                ], 500);
            }
        }

        // Create single topic (no splitting needed or disabled)
        $topic = Topic::create($validated);

        return response()->json([
            'success' => true,
            'auto_split' => false,
            'message' => 'Topic created successfully',
            'topic' => $topic,
            'id' => $topic->id
        ]);
    }

    // Update topic
    public function updateTopic(Request $request, $id)
    {
        $topic = Topic::findOrFail($id);
        
        $lesson = Lesson::findOrFail($topic->lesson_id);
        
        // Verify tutor owns the course this lesson belongs to
        $course = Course::where('created_by', Auth::id())->findOrFail($lesson->course_id);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'nullable|string',
            'video_url' => 'nullable|string',
            'order_index' => 'nullable|integer',
        ]);

        $topic->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Topic updated successfully',
            'topic' => $topic
        ]);
    }

    // Delete topic
    public function destroyTopic($id)
    {
        $topic = Topic::findOrFail($id);
        
        $lesson = Lesson::findOrFail($topic->lesson_id);
        
        // Verify tutor owns the course this lesson belongs to
        $course = Course::where('created_by', Auth::id())->findOrFail($lesson->course_id);
        
        $topic->delete();

        return response()->json([
            'success' => true,
            'message' => 'Topic deleted successfully'
        ]);
    }

    // Split an existing topic into multiple topics
    public function splitTopic($id)
    {
        $topic = Topic::with('lesson.course')->findOrFail($id);
        
        // Verify tutor owns the course
        $course = Course::where('created_by', Auth::id())->findOrFail($topic->lesson->course_id);

        $chunkWords = $course->chunk_words ?? SiteSetting::getChunkWordsDefault();

        // Check if splitting is possible
        if (!$this->topicSplitter->shouldSplit($topic->body, $chunkWords)) {
            $wordCount = $this->topicSplitter->getWordCount($topic->body);
            return response()->json([
                'success' => false,
                'message' => "Topic content is too short to split. Word count: {$wordCount}, Minimum required: " . ceil($chunkWords * 1.5)
            ], 400);
        }

        try {
            // Perform the split
            $topicIds = $this->topicSplitter->splitExistingTopic($topic, $chunkWords);

            return response()->json([
                'success' => true,
                'message' => 'Topic split successfully',
                'topics_created' => count($topicIds),
                'topic_ids' => $topicIds
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error splitting topic: ' . $e->getMessage()
            ], 500);
        }
    }

    // Preview topic split without actually creating topics
    public function previewTopicSplit($id)
    {
        $topic = Topic::with('lesson.course')->findOrFail($id);
        
        // Verify tutor owns the course
        $course = Course::where('created_by', Auth::id())->findOrFail($topic->lesson->course_id);

        $chunkWords = $course->chunk_words ?? SiteSetting::getChunkWordsDefault();
        $wordCount = $this->topicSplitter->getWordCount($topic->body);
        $canSplit = $this->topicSplitter->shouldSplit($topic->body, $chunkWords);
        $topicCount = $this->topicSplitter->calculateTopicCount($topic->body, $chunkWords);

        return response()->json([
            'success' => true,
            'can_split' => $canSplit,
            'word_count' => $wordCount,
            'chunk_words' => $chunkWords,
            'topics_would_create' => $topicCount,
            'message' => $canSplit 
                ? "This topic would be split into {$topicCount} topics" 
                : "Topic content is too short to split"
        ]);
    }

    // Get all students with their progress
    public function getStudents()
    {
        // Get tutor's courses
        $tutorCourseIds = Course::where('created_by', Auth::id())->pluck('id');
        
        // Get lessons from tutor's courses
        $tutorLessonIds = Lesson::whereIn('course_id', $tutorCourseIds)->pluck('id');
        
        // Get topics from tutor's lessons
        $tutorTopicIds = Topic::whereIn('lesson_id', $tutorLessonIds)->pluck('id');
        
        // Get students who have progress in tutor's topics
        $studentIds = UserProgress::whereIn('topic_id', $tutorTopicIds)
            ->distinct()
            ->pluck('user_id');
        
        $students = User::role('user')
            ->whereIn('id', $studentIds)
            ->with(['progress.topic.lesson.course'])
            ->get()
            ->map(function($student) use ($tutorTopicIds) {
                // Only count progress in tutor's courses
                $totalTopics = UserProgress::where('user_id', $student->id)
                    ->whereIn('topic_id', $tutorTopicIds)
                    ->count();
                $completedTopics = UserProgress::where('user_id', $student->id)
                    ->whereIn('topic_id', $tutorTopicIds)
                    ->where('completed', true)
                    ->count();
                
                $student->total_progress = $totalTopics;
                $student->completed_progress = $completedTopics;
                $student->progress_percentage = $totalTopics > 0 
                    ? round(($completedTopics / $totalTopics) * 100, 1) 
                    : 0;
                
                return $student;
            });

        return response()->json($students);
    }

    // Get single student details with course progress
    public function getStudentDetails($id)
    {
        // Get tutor's courses
        $tutorCourseIds = Course::where('created_by', Auth::id())->pluck('id');
        
        // Get lessons from tutor's courses
        $tutorLessonIds = Lesson::whereIn('course_id', $tutorCourseIds)->pluck('id');
        
        // Get topics from tutor's lessons
        $tutorTopicIds = Topic::whereIn('lesson_id', $tutorLessonIds)->pluck('id');
        
        $student = User::with([
            'progress' => function($query) use ($tutorTopicIds) {
                $query->whereIn('topic_id', $tutorTopicIds)->with('topic.lesson.course');
            },
            'lessonProgress' => function($query) use ($tutorLessonIds) {
                $query->whereIn('lesson_id', $tutorLessonIds)->with('lesson.course');
            }
        ])->findOrFail($id);

        // Calculate progress (only in tutor's courses)
        $totalTopics = $student->progress->count();
        $completedTopics = $student->progress->where('completed', true)->count();
        
        $student->total_progress = $totalTopics;
        $student->completed_progress = $completedTopics;
        $student->progress_percentage = $totalTopics > 0 
            ? round(($completedTopics / $totalTopics) * 100, 1) 
            : 0;

        // Group progress by course
        $courseProgress = [];
        foreach ($student->lessonProgress as $lessonProgress) {
            $courseName = $lessonProgress->lesson->course->name ?? 'Unknown';
            if (!isset($courseProgress[$courseName])) {
                $courseProgress[$courseName] = [
                    'total' => 0,
                    'completed' => 0
                ];
            }
            $courseProgress[$courseName]['total']++;
            if ($lessonProgress->completed) {
                $courseProgress[$courseName]['completed']++;
            }
        }

        $student->course_progress = $courseProgress;

        return response()->json($student);
    }

    // Get quiz questions for a lesson
    public function getQuizQuestions($lessonId)
    {
        $lesson = Lesson::findOrFail($lessonId);
        
        // Verify tutor owns the course this lesson belongs to
        $course = Course::where('created_by', Auth::id())->findOrFail($lesson->course_id);
        
        $questions = QuizQuestion::where('lesson_id', $lessonId)
            ->orderBy('order_index')
            ->orderBy('id')
            ->get()
            ->map(function ($question) {
                $options = json_decode($question->options_json, true);
                if (!is_array($options)) {
                    $options = [];
                }

                // Use correct_index if correct_answer is empty
                $correctAnswer = !empty($question->correct_answer) ? $question->correct_answer : $question->correct_index;

                return [
                    'id' => $question->id,
                    'lesson_id' => $question->lesson_id,
                    'question' => $question->question,
                    'options' => $options,
                    'correct_answer' => $correctAnswer,
                    'correct_index' => $question->correct_index,
                    'explanation' => $question->explanation,
                    'order_index' => $question->order_index,
                    'created_at' => $question->created_at,
                    'updated_at' => $question->updated_at,
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $questions,
            'questions' => $questions
        ]);
    }

    // Create quiz question
    public function storeQuizQuestion(Request $request)
    {
        $validated = $request->validate([
            'lesson_id' => 'required|exists:lessons,id',
            'question' => 'required|string',
            'explanation' => 'nullable|string',
            'options' => 'required|array|min:2',
            'options.*.text' => 'required_with:options|string',
            'options.*.correct' => 'nullable|boolean',
        ]);

        $lesson = Lesson::findOrFail($validated['lesson_id']);
        
        // Verify tutor owns the course this lesson belongs to
        $course = Course::where('created_by', Auth::id())->findOrFail($lesson->course_id);

        // Extract options and find correct index
        $options = collect($request->input('options', []))->map(function ($opt) {
            return $opt['text'] ?? '';
        })->filter(fn($text) => !empty($text))->values()->toArray();

        $correctIndex = 0;
        foreach ($request->input('options', []) as $idx => $opt) {
            if (!empty($opt['correct']) && !empty($opt['text'])) {
                $correctIndex = $idx;
                break;
            }
        }

        // Set default order_index
        $maxOrder = QuizQuestion::where('lesson_id', $validated['lesson_id'])->max('order_index') ?? 0;

        $question = QuizQuestion::create([
            'lesson_id' => $validated['lesson_id'],
            'question' => $validated['question'],
            'options_json' => json_encode($options),
            'correct_index' => $correctIndex,
            'explanation' => $request->input('explanation'),
            'order_index' => $maxOrder + 1,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Quiz question created successfully',
            'question' => $question
        ]);
    }

    // Update quiz question
    public function updateQuizQuestion(Request $request, $id)
    {
        $question = QuizQuestion::findOrFail($id);
        
        $lesson = Lesson::findOrFail($question->lesson_id);
        
        // Verify tutor owns the course this lesson belongs to
        $course = Course::where('created_by', Auth::id())->findOrFail($lesson->course_id);

        $validated = $request->validate([
            'question' => 'required|string',
            'explanation' => 'nullable|string',
            'options' => 'required|array|min:2',
            'options.*.text' => 'required_with:options|string',
            'options.*.correct' => 'nullable|boolean',
        ]);

        // Extract options and find correct index
        $options = collect($request->input('options', []))->map(function ($opt) {
            return $opt['text'] ?? '';
        })->filter(fn($text) => !empty($text))->values()->toArray();

        $correctIndex = 0;
        foreach ($request->input('options', []) as $idx => $opt) {
            if (!empty($opt['correct']) && !empty($opt['text'])) {
                $correctIndex = $idx;
                break;
            }
        }

        $question->update([
            'question' => $validated['question'],
            'options_json' => json_encode($options),
            'correct_index' => $correctIndex,
            'explanation' => $request->input('explanation'),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Quiz question updated successfully',
            'question' => $question
        ]);
    }

    // Delete quiz question
    public function destroyQuizQuestion($id)
    {
        $question = QuizQuestion::findOrFail($id);
        
        $lesson = Lesson::findOrFail($question->lesson_id);
        
        // Verify tutor owns the course this lesson belongs to
        $course = Course::where('created_by', Auth::id())->findOrFail($lesson->course_id);
        
        $question->delete();

        return response()->json([
            'success' => true,
            'message' => 'Quiz question deleted successfully'
        ]);
    }

    /**
     * Get single lesson details
     */
    public function getLesson(Lesson $lesson)
    {
        // Verify tutor owns the course this lesson belongs to
        $course = Course::where('created_by', Auth::id())->findOrFail($lesson->course_id);

        return response()->json([
            'success' => true,
            'lesson' => $lesson
        ]);
    }

    /**
     * Get single topic details
     */
    public function getTopic(Topic $topic)
    {
        // Verify tutor owns the course this topic belongs to
        $lesson = Lesson::findOrFail($topic->lesson_id);
        $course = Course::where('created_by', Auth::id())->findOrFail($lesson->course_id);

        return response()->json([
            'success' => true,
            'topic' => $topic
        ]);
    }

    /**
     * Get single quiz question details
     */
    public function getQuizQuestion(QuizQuestion $quizQuestion)
    {
        // Verify tutor owns the course this quiz question belongs to
        $lesson = Lesson::findOrFail($quizQuestion->lesson_id);
        $course = Course::where('created_by', Auth::id())->findOrFail($lesson->course_id);

        $options = json_decode($quizQuestion->options_json, true);
        if (!is_array($options)) {
            $options = [];
        }

        // Use correct_index if correct_answer is empty
        $correctAnswer = !empty($quizQuestion->correct_answer) ? $quizQuestion->correct_answer : $quizQuestion->correct_index;

        return response()->json([
            'success' => true,
            'question' => [
                'id' => $quizQuestion->id,
                'lesson_id' => $quizQuestion->lesson_id,
                'question' => $quizQuestion->question,
                'options' => $options,
                'correct_answer' => $correctAnswer,
                'correct_index' => $quizQuestion->correct_index,
                'explanation' => $quizQuestion->explanation,
                'order_index' => $quizQuestion->order_index,
                'created_at' => $quizQuestion->created_at,
                'updated_at' => $quizQuestion->updated_at,
            ],
            'data' => [
                'id' => $quizQuestion->id,
                'lesson_id' => $quizQuestion->lesson_id,
                'question' => $quizQuestion->question,
                'options' => $options,
                'correct_answer' => $correctAnswer,
                'correct_index' => $quizQuestion->correct_index,
                'explanation' => $quizQuestion->explanation,
                'order_index' => $quizQuestion->order_index,
                'created_at' => $quizQuestion->created_at,
                'updated_at' => $quizQuestion->updated_at,
            ]
        ]);
    }

    /**
     * Get available courses for prerequisites (only tutor's courses)
     */
    public function getAvailablePrerequisites($excludeCourseId = null)
    {
        try {
            // Get only tutor's courses for prerequisites
            $courses = Course::where('created_by', Auth::id())
                ->when($excludeCourseId, function ($query) use ($excludeCourseId) {
                    return $query->where('id', '!=', $excludeCourseId);
                })
                ->select('id', 'name', 'slug')
                ->orderBy('name')
                ->get();

            return response()->json([
                'success' => true,
                'courses' => $courses
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching available prerequisites: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get prerequisites for a course
     */
    public function getCoursePrerequisites($courseId)
    {
        try {
            // Verify tutor owns the course
            $course = Course::where('created_by', Auth::id())->findOrFail($courseId);
            
            $prerequisites = $this->courseService->getPrerequisites($courseId);

            return response()->json([
                'success' => true,
                'prerequisites' => $prerequisites
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching prerequisites: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get course statistics
     */
    public function getCourseStats($courseId)
    {
        try {
            // Verify tutor owns the course
            $course = Course::where('created_by', Auth::id())->findOrFail($courseId);
            
            $stats = $this->courseService->getCourseStats($courseId);

            if (!$stats) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'stats' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching course stats: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reorder lessons
     */
    public function reorderLessons(Request $request)
    {
        $request->validate([
            'lesson_ids' => 'sometimes|array',
            'lesson_ids.*' => 'exists:lessons,id',
            'order' => 'sometimes|array',
            'order.*.id' => 'required_with:order|exists:lessons,id',
            'order.*.position' => 'nullable|integer',
            'course_id' => 'required|exists:courses,id',
        ]);

        $lessonIds = $request->lesson_ids ?? collect($request->order ?? [])->pluck('id')->toArray();
        if (empty($lessonIds)) {
            return response()->json(['success' => false, 'message' => 'lesson_ids or order is required'], 422);
        }

        // Verify tutor owns all lessons
        foreach ($lessonIds as $lessonId) {
            $lesson = Lesson::findOrFail($lessonId);
            Course::where('created_by', Auth::id())->where('id', $lesson->course_id)->findOrFail();
        }

        $orderMap = collect($request->order ?? [])->keyBy('id');
        foreach (array_values($lessonIds) as $index => $lessonId) {
            $position = $orderMap[$lessonId]['position'] ?? $index;
            Lesson::where('id', $lessonId)->update(['order_index' => $position]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Lessons reordered successfully'
        ]);
    }

    /**
     * Reorder topics
     */
    public function reorderTopics(Request $request)
    {
        $request->validate([
            'topic_ids' => 'sometimes|array',
            'topic_ids.*' => 'exists:topics,id',
            'order' => 'sometimes|array',
            'order.*.id' => 'required_with:order|exists:topics,id',
            'order.*.position' => 'nullable|integer',
            'lesson_id' => 'required|exists:lessons,id',
        ]);

        $topicIds = $request->topic_ids ?? collect($request->order ?? [])->pluck('id')->toArray();
        if (empty($topicIds)) {
            return response()->json(['success' => false, 'message' => 'topic_ids or order is required'], 422);
        }

        // Verify tutor owns all topics
        foreach ($topicIds as $topicId) {
            $topic = Topic::findOrFail($topicId);
            $lesson = Lesson::findOrFail($topic->lesson_id);
            Course::where('created_by', Auth::id())->where('id', $lesson->course_id)->findOrFail();
        }

        $orderMap = collect($request->order ?? [])->keyBy('id');
        foreach (array_values($topicIds) as $index => $topicId) {
            $position = $orderMap[$topicId]['position'] ?? $index;
            Topic::where('id', $topicId)->update(['order_index' => $position]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Topics reordered successfully'
        ]);
    }

    protected function deleteStoredFile(?string $url): void
    {
        if (!$url) {
            return;
        }

        $prefix = '/storage/';
        if (str_starts_with($url, $prefix)) {
            $relativePath = substr($url, strlen($prefix));
            Storage::disk('public')->delete($relativePath);
        }
    }

    /**
     * Get certificate templates for dropdowns
     */
    public function getCertificateTemplates()
    {
        $templates = \App\Models\CertificateTemplate::where('is_active', true)
            ->orderBy('name')
            ->get(['id', 'name', 'description']);

        return response()->json([
            'success' => true,
            'templates' => $templates
        ]);
    }
}
