<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class CourseEnrollment extends Model
{
    protected $fillable = [
        'user_id',
        'course_id',
        'status',
        'enrolled_at',
        'completed_at',
        'progress',
    ];

    protected $casts = [
        'enrolled_at' => 'datetime',
        'completed_at' => 'datetime',
        'progress' => 'decimal:2',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function course(): BelongsTo
    {
        return $this->belongsTo(Course::class);
    }

    /**
     * Check if enrollment is active
     */
    public function isActive(): bool
    {
        return in_array($this->status, ['enrolled', 'in-progress']);
    }

    /**
     * Mark enrollment as in progress
     */
    public function markAsInProgress(): void
    {
        if ($this->status === 'enrolled') {
            $this->update(['status' => 'in-progress']);
        }
    }

    /**
     * Mark enrollment as completed
     */
    public function markAsCompleted(): void
    {
        $this->update([
            'status' => 'completed',
            'completed_at' => now(),
            'progress' => 100,
        ]);
    }

    /**
     * Update progress percentage
     */
    public function updateProgress(float $progress): void
    {
        $this->update([
            'progress' => min($progress, 100),
            'status' => $progress > 0 ? 'in-progress' : 'enrolled',
        ]);
    }
}
