<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class SiteSetting extends Model
{
    protected $fillable = [
        'key',
        'value',
        'type',
        'group',
        'description'
    ];

    /**
     * Get a setting value by key (with caching)
     */
    public static function get(string $key, $default = null)
    {
        return Cache::remember("setting_{$key}", 3600, function () use ($key, $default) {
            $setting = self::where('key', $key)->first();
            return $setting ? $setting->value : $default;
        });
    }

    /**
     * Set a setting value
     */
    public static function set(string $key, $value, string $type = 'text', string $group = 'general', string $description = null)
    {
        $setting = self::updateOrCreate(
            ['key' => $key],
            [
                'value' => $value,
                'type' => $type,
                'group' => $group,
                'description' => $description
            ]
        );

        // Clear cache
        Cache::forget("setting_{$key}");

        return $setting;
    }

    /**
     * Get all settings grouped by category
     */
    public static function getGrouped()
    {
        return self::all()->groupBy('group');
    }

    /**
     * Get all settings as key-value array
     */
    public static function getAllAsArray()
    {
        return Cache::remember('all_settings', 3600, function () {
            return self::pluck('value', 'key')->toArray();
        });
    }

    /**
     * Clear all settings cache
     */
    public static function clearCache()
    {
        Cache::forget('all_settings');
        self::all()->each(function ($setting) {
            Cache::forget("setting_{$setting->key}");
        });
    }

    /**
     * Get chunk words default setting
     */
    public static function getChunkWordsDefault()
    {
        return (int) self::get('chunk_words_default', 150);
    }

    /**
     * Get chunk words minimum setting
     */
    public static function getChunkWordsMin()
    {
        return (int) self::get('chunk_words_min', 50);
    }

    /**
     * Get chunk words maximum setting
     */
    public static function getChunkWordsMax()
    {
        return (int) self::get('chunk_words_max', 500);
    }

    /**
     * Get all chunk words settings
     */
    public static function getChunkWordsSettings()
    {
        return [
            'default' => self::getChunkWordsDefault(),
            'min' => self::getChunkWordsMin(),
            'max' => self::getChunkWordsMax()
        ];
    }
}
