<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable
{
    use HasFactory, Notifiable, HasRoles;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'is_admin',
        'phone',
        'location',
        'bio',
        'avatar',
        'email_notifications',
        'course_updates',
        'achievement_notifications',
        'weekly_digest',
        'profile_visible',
        'show_leaderboard',
        'share_progress',
        'language',
        'timezone',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'is_admin' => 'boolean',
        ];
    }

    public function progress()
    {
        return $this->hasMany(UserProgress::class);
    }

    public function lessonProgress()
    {
        return $this->hasMany(UserLessonProgress::class);
    }

    public function accessCodesUsed()
    {
        return $this->hasMany(AccessCode::class, 'used_by');
    }

    public function accessCodesCreated()
    {
        return $this->hasMany(AccessCode::class, 'created_by');
    }

    /**
     * Certificates earned by this user
     */
    public function certificates()
    {
        return $this->hasMany(IssuedCertificate::class);
    }

    /**
     * Course enrollments
     */
    public function enrollments()
    {
        return $this->hasMany(CourseEnrollment::class);
    }

    /**
     * Enrolled courses
     */
    public function enrolledCourses()
    {
        return $this->belongsToMany(Course::class, 'course_enrollments', 'user_id', 'course_id')
            ->withPivot('status', 'enrolled_at', 'completed_at', 'progress')
            ->withTimestamps();
    }

    /**
     * Badges earned by this user
     */
    public function badges()
    {
        return $this->belongsToMany(Badge::class, 'user_badges')
            ->withPivot('earned_at', 'metadata')
            ->withTimestamps();
    }

    /**
     * Check if user has earned a specific badge
     */
    public function hasBadge($badgeId): bool
    {
        return $this->badges()->where('badge_id', $badgeId)->exists();
    }

    /**
     * Award a badge to this user
     */
    public function awardBadge($badgeId, array $metadata = []): void
    {
        if (!$this->hasBadge($badgeId)) {
            $this->badges()->attach($badgeId, [
                'earned_at' => now(),
                'metadata' => json_encode($metadata),
            ]);
        }
    }

    /**
     * Get total badge points
     */
    public function getTotalBadgePoints(): int
    {
        return $this->badges()->sum('points');
    }

    /**
     * Penalty cards issued to this user
     */
    public function penaltyCards()
    {
        return $this->hasMany(UserPenaltyCard::class);
    }

    /**
     * Active penalty cards
     */
    public function activePenaltyCards()
    {
        return $this->hasMany(UserPenaltyCard::class)->where('is_active', true);
    }

    /**
     * Penalty history for this user
     */
    public function penaltyHistory()
    {
        return $this->hasMany(PenaltyHistory::class);
    }

    /**
     * Activity logs for this user
     */
    public function activityLogs()
    {
        return $this->hasMany(UserActivityLog::class);
    }

    /**
     * Check if user has active yellow card
     */
    public function hasActiveYellowCard(): bool
    {
        return $this->activePenaltyCards()->where('card_type', 'yellow')->exists();
    }

    /**
     * Check if user has active red card
     */
    public function hasActiveRedCard(): bool
    {
        return $this->activePenaltyCards()->where('card_type', 'red')->exists();
    }

    /**
     * Get count of active penalty cards
     */
    public function getActivePenaltyCardCount(): int
    {
        return $this->activePenaltyCards()->count();
    }

    /**
     * Get total points deducted from penalties
     */
    public function getTotalPenaltyPoints(): int
    {
        return $this->penaltyCards()->sum('points_deducted');
    }

    /**
     * Get last activity timestamp
     */
    public function getLastActivityAt()
    {
        $lastLog = $this->activityLogs()->orderBy('activity_at', 'desc')->first();
        return $lastLog ? $lastLog->activity_at : null;
    }

    /**
     * Get days since last activity
     */
    public function getDaysSinceLastActivity(): ?int
    {
        $lastActivity = $this->getLastActivityAt();
        return $lastActivity ? $lastActivity->diffInDays(now()) : null;
    }

    /**
     * Log user activity
     */
    public function logActivity(string $activityType, $relatedId = null, string $relatedType = null, array $metadata = []): void
    {
        UserActivityLog::create([
            'user_id' => $this->id,
            'activity_type' => $activityType,
            'related_id' => $relatedId,
            'related_type' => $relatedType,
            'activity_at' => now(),
            'metadata' => $metadata
        ]);
    }
}