<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class UserPenaltyCard extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'penalty_rule_id',
        'issued_by',
        'card_type',
        'reason',
        'points_deducted',
        'issued_at',
        'expires_at',
        'escalated_at',
        'is_active',
        'revoked_at',
        'revoked_by',
        'revoke_reason',
        'metadata'
    ];

    protected $casts = [
        'user_id' => 'integer',
        'penalty_rule_id' => 'integer',
        'issued_by' => 'integer',
        'points_deducted' => 'integer',
        'issued_at' => 'datetime',
        'expires_at' => 'datetime',
        'escalated_at' => 'datetime',
        'is_active' => 'boolean',
        'revoked_at' => 'datetime',
        'revoked_by' => 'integer',
        'metadata' => 'array'
    ];

    /**
     * Get the user who received the penalty card
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the penalty rule that triggered this card
     */
    public function penaltyRule()
    {
        return $this->belongsTo(PenaltyRule::class);
    }

    /**
     * Get the admin/tutor who issued the card
     */
    public function issuer()
    {
        return $this->belongsTo(User::class, 'issued_by');
    }

    /**
     * Get the user who revoked the card
     */
    public function revoker()
    {
        return $this->belongsTo(User::class, 'revoked_by');
    }

    /**
     * Get penalty history related to this card
     */
    public function history()
    {
        return $this->hasMany(PenaltyHistory::class, 'penalty_card_id');
    }

    /**
     * Scope for active cards
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for specific card type
     */
    public function scopeCardType($query, $type)
    {
        return $query->where('card_type', $type);
    }

    /**
     * Scope for yellow cards
     */
    public function scopeYellowCards($query)
    {
        return $query->where('card_type', 'yellow');
    }

    /**
     * Scope for red cards
     */
    public function scopeRedCards($query)
    {
        return $query->where('card_type', 'red');
    }

    /**
     * Scope for cards issued by auto-system
     */
    public function scopeAutoIssued($query)
    {
        return $query->whereNull('issued_by');
    }

    /**
     * Scope for cards issued manually
     */
    public function scopeManuallyIssued($query)
    {
        return $query->whereNotNull('issued_by');
    }

    /**
     * Check if card is yellow
     */
    public function isYellow()
    {
        return $this->card_type === 'yellow';
    }

    /**
     * Check if card is red
     */
    public function isRed()
    {
        return $this->card_type === 'red';
    }

    /**
     * Check if card is expired
     */
    public function isExpired()
    {
        return $this->expires_at && $this->expires_at->isPast();
    }

    /**
     * Check if card can be escalated
     */
    public function canEscalate()
    {
        return $this->is_active 
            && $this->card_type === 'yellow' 
            && !$this->escalated_at
            && $this->penaltyRule 
            && $this->penaltyRule->escalation_days;
    }

    /**
     * Get days since card was issued
     */
    public function daysSinceIssued()
    {
        return $this->issued_at->diffInDays(now());
    }

    /**
     * Get formatted card color for UI
     */
    public function getCardColorAttribute()
    {
        return [
            'yellow' => '#f59e0b',
            'red' => '#ef4444'
        ][$this->card_type] ?? '#6b7280';
    }

    /**
     * Get formatted card icon
     */
    public function getCardIconAttribute()
    {
        return [
            'yellow' => '⚠️',
            'red' => '🚫'
        ][$this->card_type] ?? '⚪';
    }
}