<?php

namespace App\Services;

use App\Models\Badge;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class BadgeService
{
    /**
     * Check and award badges for a user based on their activity
     */
    public function checkAndAwardBadges(User $user, string $trigger, array $context = []): array
    {
        $awardedBadges = [];

        $badges = Badge::active()
            ->where('category', $trigger)
            ->whereDoesntHave('users', function ($query) use ($user) {
                $query->where('user_id', $user->id);
            })
            ->get();

        foreach ($badges as $badge) {
            if ($this->checkCriteria($user, $badge, $context)) {
                $this->awardBadge($user, $badge, $context);
                $awardedBadges[] = $badge;
            }
        }

        return $awardedBadges;
    }

    /**
     * Award a badge to a user
     */
    public function awardBadge(User $user, Badge $badge, array $metadata = []): void
    {
        if (!$user->hasBadge($badge->id)) {
            $user->badges()->attach($badge->id, [
                'earned_at' => now(),
                'metadata' => json_encode($metadata),
            ]);

            Log::info("Badge awarded", [
                'user_id' => $user->id,
                'badge_id' => $badge->id,
                'badge_name' => $badge->name,
            ]);
        }
    }

    /**
     * Check if user meets badge criteria
     */
    protected function checkCriteria(User $user, Badge $badge, array $context): bool
    {
        $criteria = $badge->criteria ?? [];

        if (empty($criteria)) {
            return false;
        }

        $type = $criteria['type'] ?? null;

        switch ($type) {
            case 'course_completion':
                return $this->checkCourseCompletion($user, $criteria, $context);

            case 'quiz_score':
                return $this->checkQuizScore($user, $criteria, $context);

            case 'perfect_score':
                return $this->checkPerfectScore($user, $criteria, $context);

            case 'course_count':
                return $this->checkCourseCount($user, $criteria);

            case 'speed_completion':
                return $this->checkSpeedCompletion($user, $criteria, $context);

            case 'streak':
                return $this->checkStreak($user, $criteria);

            case 'lesson_count':
                return $this->checkLessonCount($user, $criteria);

            default:
                return false;
        }
    }

    /**
     * Check course completion criteria
     */
    protected function checkCourseCompletion(User $user, array $criteria, array $context): bool
    {
        // For "First Course Completed" badge
        if (isset($criteria['first_course']) && $criteria['first_course']) {
            $completedCourses = DB::table('user_progress')
                ->where('user_id', $user->id)
                ->where('status', 'completed')
                ->count();
            
            return $completedCourses == 1;
        }

        return isset($context['course_completed']) && $context['course_completed'];
    }

    /**
     * Check quiz score criteria
     */
    protected function checkQuizScore(User $user, array $criteria, array $context): bool
    {
        $requiredScore = $criteria['min_score'] ?? 80;
        $currentScore = $context['quiz_score'] ?? 0;

        return $currentScore >= $requiredScore;
    }

    /**
     * Check perfect score criteria
     */
    protected function checkPerfectScore(User $user, array $criteria, array $context): bool
    {
        return isset($context['quiz_score']) && $context['quiz_score'] == 100;
    }

    /**
     * Check course count criteria
     */
    protected function checkCourseCount(User $user, array $criteria): bool
    {
        $requiredCount = $criteria['required_count'] ?? 1;
        
        $completedCourses = DB::table('user_progress')
            ->where('user_id', $user->id)
            ->where('status', 'completed')
            ->count();

        return $completedCourses >= $requiredCount;
    }

    /**
     * Check speed completion criteria (complete course in X hours)
     */
    protected function checkSpeedCompletion(User $user, array $criteria, array $context): bool
    {
        $maxHours = $criteria['max_hours'] ?? 24;
        $startTime = $context['course_start_time'] ?? null;
        $endTime = now();

        if (!$startTime) {
            return false;
        }

        $hoursTaken = $startTime->diffInHours($endTime);
        return $hoursTaken <= $maxHours;
    }

    /**
     * Check login streak criteria
     */
    protected function checkStreak(User $user, array $criteria): bool
    {
        $requiredDays = $criteria['required_days'] ?? 7;
        
        // This would require a separate user_logins table to track
        // For now, return false
        return false;
    }

    /**
     * Check lesson count criteria
     */
    protected function checkLessonCount(User $user, array $criteria): bool
    {
        $requiredCount = $criteria['required_count'] ?? 1;
        
        $completedLessons = DB::table('user_lesson_progress')
            ->where('user_id', $user->id)
            ->where('is_completed', true)
            ->count();

        return $completedLessons >= $requiredCount;
    }

    /**
     * Get all badges for display
     */
    public function getAllBadgesWithUserStatus(User $user)
    {
        $badges = Badge::active()->orderBy('order')->orderBy('points')->get();

        return $badges->map(function ($badge) use ($user) {
            $userBadge = $user->badges()->where('badge_id', $badge->id)->first();
            
            return [
                'id' => $badge->id,
                'name' => $badge->name,
                'description' => $badge->description,
                'icon' => $badge->icon,
                'type' => $badge->type,
                'category' => $badge->category,
                'points' => $badge->points,
                'color' => $badge->color,
                'earned' => $userBadge !== null,
                'earned_at' => $userBadge ? $userBadge->pivot->earned_at : null,
                'progress' => $this->calculateBadgeProgress($user, $badge),
            ];
        });
    }

    /**
     * Calculate progress towards earning a badge
     */
    protected function calculateBadgeProgress(User $user, Badge $badge): ?array
    {
        if ($user->hasBadge($badge->id)) {
            return ['current' => 100, 'required' => 100, 'percentage' => 100];
        }

        $criteria = $badge->criteria ?? [];
        $type = $criteria['type'] ?? null;

        switch ($type) {
            case 'course_count':
                $required = $criteria['required_count'] ?? 1;
                $current = DB::table('user_progress')
                    ->where('user_id', $user->id)
                    ->where('status', 'completed')
                    ->count();
                return [
                    'current' => $current,
                    'required' => $required,
                    'percentage' => min(100, ($current / $required) * 100)
                ];

            case 'lesson_count':
                $required = $criteria['required_count'] ?? 1;
                $current = DB::table('user_lesson_progress')
                    ->where('user_id', $user->id)
                    ->where('is_completed', true)
                    ->count();
                return [
                    'current' => $current,
                    'required' => $required,
                    'percentage' => min(100, ($current / $required) * 100)
                ];

            default:
                return null;
        }
    }

    /**
     * Get user's badge statistics
     */
    public function getUserBadgeStats(User $user): array
    {
        $totalBadges = Badge::active()->count();
        $earnedBadges = $user->badges()->count();
        $totalPoints = $user->getTotalBadgePoints();

        return [
            'total_badges' => $totalBadges,
            'earned_badges' => $earnedBadges,
            'pending_badges' => $totalBadges - $earnedBadges,
            'total_points' => $totalPoints,
            'completion_percentage' => $totalBadges > 0 ? round(($earnedBadges / $totalBadges) * 100) : 0,
        ];
    }
}
