<?php

namespace App\Services;

use App\Models\CertificateTemplate;
use App\Models\IssuedCertificate;
use App\Models\User;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use SimpleSoftwareIO\QrCode\Facades\QrCode;

class CertificateGeneratorService
{
    /**
     * Generate a certificate for a user
     *
     * @param User $user
     * @param CertificateTemplate $template
     * @param mixed $assignable (Course or Lesson)
     * @return IssuedCertificate
     */
    public function generateCertificate(User $user, CertificateTemplate $template, $assignable)
    {
        // Generate unique certificate code
        $certificateCode = $this->generateUniqueCertificateCode();

        // Generate verification URL
        $verificationUrl = route('certificate.verify', ['code' => $certificateCode]);

        // Generate QR code if enabled
        $qrCodePath = null;
        if ($template->qr_code_enabled) {
            $qrCodePath = $this->generateQRCode($verificationUrl, $certificateCode);
        }

        // Create issued certificate record
        $issuedCertificate = IssuedCertificate::create([
            'user_id' => $user->id,
            'certificate_template_id' => $template->id,
            'assignable_type' => get_class($assignable),
            'assignable_id' => $assignable->id,
            'certificate_code' => $certificateCode,
            'qr_code_path' => $qrCodePath,
            'verification_url' => $verificationUrl,
            'issued_at' => now(),
        ]);

        // Generate the certificate HTML/Image (PDF generation will be added next)
        $certificatePath = $this->generateCertificateImage($issuedCertificate, $template, $user, $assignable);
        
        // Update the certificate with the PDF path
        $issuedCertificate->update(['pdf_path' => $certificatePath]);

        return $issuedCertificate;
    }

    /**
     * Generate a unique certificate code
     *
     * @return string
     */
    protected function generateUniqueCertificateCode(): string
    {
        do {
            // Format: CERT-XXXX-XXXX-XXXX
            $code = 'CERT-' . strtoupper(Str::random(4)) . '-' . strtoupper(Str::random(4)) . '-' . strtoupper(Str::random(4));
        } while (IssuedCertificate::where('certificate_code', $code)->exists());

        return $code;
    }

    /**
     * Generate QR code for certificate verification
     *
     * @param string $url
     * @param string $certificateCode
     * @return string|null
     */
    protected function generateQRCode(string $url, string $certificateCode): ?string
    {
        try {
            // Create QR code directory if it doesn't exist
            $qrDir = storage_path('app/public/certificates/qrcodes');
            if (!file_exists($qrDir)) {
                mkdir($qrDir, 0755, true);
            }

            // Generate QR code filename
            $filename = 'qr_' . $certificateCode . '.png';
            $path = $qrDir . '/' . $filename;

            // Generate QR code image
            QrCode::format('png')
                ->size(300)
                ->margin(1)
                ->generate($url, $path);

            // Return relative path for storage
            return 'certificates/qrcodes/' . $filename;

        } catch (\Exception $e) {
            Log::error('QR Code generation failed: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Generate certificate image/HTML
     * This is a placeholder - will be expanded for full PDF generation
     *
     * @param IssuedCertificate $certificate
     * @param CertificateTemplate $template
     * @param User $user
     * @param mixed $assignable
     * @return string
     */
    protected function generateCertificateImage($certificate, $template, $user, $assignable): string
    {
        // For now, we'll create an HTML file
        // Later this can be converted to PDF using packages like DOMPDF or mPDF
        
        $html = $this->buildCertificateHTML($certificate, $template, $user, $assignable);
        
        // Save HTML certificate
        $certDir = storage_path('app/public/certificates/issued');
        if (!file_exists($certDir)) {
            mkdir($certDir, 0755, true);
        }

        $filename = 'cert_' . $certificate->certificate_code . '.html';
        $path = $certDir . '/' . $filename;
        
        file_put_contents($path, $html);

        return 'certificates/issued/' . $filename;
    }

    /**
     * Build certificate HTML
     *
     * @param IssuedCertificate $certificate
     * @param CertificateTemplate $template
     * @param User $user
     * @param mixed $assignable
     * @return string
     */
    protected function buildCertificateHTML($certificate, $template, $user, $assignable): string
    {
        $courseName = $assignable->title ?? $assignable->name ?? 'Course';
        $studentName = $user->name;
        $completionDate = $certificate->issued_at->format('F j, Y');
        $certificateCode = $certificate->certificate_code;

        // Get dimensions based on orientation
        $width = $template->orientation === 'portrait' ? '794px' : '1122px';
        $height = $template->orientation === 'portrait' ? '1122px' : '794px';

        // Get background style
        $backgroundStyle = '';
        if ($template->background_image) {
            $backgroundUrl = asset('storage/' . $template->background_image);
            $backgroundStyle = "background-image: url('$backgroundUrl'); background-size: cover; background-position: center;";
        } else {
            $backgroundStyle = 'background: white;';
        }

        // Get border config from layout_config if exists
        $borderStyle = '';
        if (isset($template->layout_config['border'])) {
            $border = $template->layout_config['border'];
            $borderStyle = "border: {$border['width']}px {$border['style']} {$border['color']};";
        }

        // Build text fields HTML
        $textFieldsHtml = '';
        $textFields = $template->text_fields ?? [];
        
        // Map field names to actual data
        $fieldData = [
            'name' => $studentName,
            'course' => $courseName,
            'date' => $completionDate,
            'code' => $certificateCode,
        ];

        foreach ($fieldData as $key => $value) {
            if (isset($textFields[$key])) {
                $field = $textFields[$key];
                $x = $field['x'] ?? 50;
                $y = $field['y'] ?? 200;
                $size = $field['size'] ?? 24;
                $font = $field['font'] ?? 'Arial';
                $color = $field['color'] ?? '#000000';
                $align = $field['align'] ?? 'center';
                $weight = $field['weight'] ?? 'normal';

                $transform = $align === 'center' ? 'translateX(-50%)' : ($align === 'right' ? 'translateX(-100%)' : 'translateX(0)');

                $textFieldsHtml .= "<div style='position: absolute; left: {$x}%; top: {$y}px; font-family: {$font}; font-size: {$size}px; color: {$color}; text-align: {$align}; transform: {$transform}; font-weight: {$weight}; white-space: nowrap;'>{$value}</div>";
            }
        }

        // Add QR code if enabled
        $qrCodeHtml = '';
        if ($template->qr_code_enabled && $certificate->qr_code_path) {
            $qrConfig = $template->qr_code_config ?? [];
            $qrX = $qrConfig['x'] ?? 85;
            $qrY = $qrConfig['y'] ?? 650;
            $qrSize = $qrConfig['size'] ?? 100;

            $qrUrl = asset('storage/' . $certificate->qr_code_path);
            $qrCodeHtml = "<img src='$qrUrl' style='position: absolute; left: {$qrX}%; top: {$qrY}px; width: {$qrSize}px; height: {$qrSize}px; transform: translateX(-50%);' alt='Verification QR Code'>";
        }

        return <<<HTML
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Certificate - {$certificateCode}</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: Arial, sans-serif; display: flex; justify-content: center; align-items: center; min-height: 100vh; background: #f5f5f5; padding: 20px; }
        .certificate { width: {$width}; height: {$height}; position: relative; {$backgroundStyle} {$borderStyle} box-shadow: 0 10px 30px rgba(0,0,0,0.2); }
    </style>
</head>
<body>
    <div class="certificate">
        {$textFieldsHtml}
        {$qrCodeHtml}
    </div>
</body>
</html>
HTML;
    }

    /**
     * Get QR code image for certificate
     *
     * @param IssuedCertificate $certificate
     * @return string|null
     */
    public function getQRCodePath(IssuedCertificate $certificate): ?string
    {
        if ($certificate->qr_code_path && Storage::disk('public')->exists($certificate->qr_code_path)) {
            return asset('storage/' . $certificate->qr_code_path);
        }

        return null;
    }

    /**
     * Build a preview HTML for a template using sample data.
     *
     * @param CertificateTemplate $template
     * @return string
     */
    public function buildPreviewHTML(CertificateTemplate $template): string
    {
        $code = 'PREVIEW-' . strtoupper(Str::random(6));

        // Prepare lightweight objects compatible with build method
        $certificate = (object) [
            'certificate_code' => $code,
            'issued_at' => now(),
            'qr_code_path' => null,
        ];

        // Optionally generate a QR for preview
        if ($template->qr_code_enabled) {
            try {
                $url = route('certificate.verify', ['code' => $code]);
                $path = $this->generateQRCode($url, $code);
                $certificate->qr_code_path = $path;
            } catch (\Exception $e) {
                // Ignore QR failures in preview
            }
        }

        $user = (object) ['name' => 'John Doe'];
        $assignable = (object) ['title' => 'Sample Course'];

        return $this->buildCertificateHTML($certificate, $template, $user, $assignable);
    }
}
