<?php

namespace App\Services;

use App\Models\NotificationSetting;
use App\Models\User;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class NotificationService
{
    protected $settings;

    public function __construct()
    {
        $this->settings = NotificationSetting::getSettings();
    }

    public function send(User $user, string $title, string $message, array $data = [])
    {
        $results = [];
        if ($this->settings->email_enabled) {
            $results['email'] = $this->sendEmail($user, $title, $message, $data);
        }
        if ($this->settings->push_enabled && $user->device_token) {
            $results['push'] = $this->sendPushNotification($user, $title, $message, $data);
        }
        if ($this->settings->sms_enabled && $user->phone) {
            $results['sms'] = $this->sendSms($user, $title, $message);
        }
        return $results;
    }

    public function sendEmail(User $user, string $subject, mixed $message, array $data = [])
    {
        try {
            if (!$this->settings->email_enabled) {
                Log::warning('Email send skipped because notifications are disabled', [
                    'user' => $user->email,
                ]);

                return [
                    'success' => false,
                    'message' => 'Email notifications are currently disabled.',
                ];
            }

            $mailer = $this->applyMailConfig();
            $content = $this->normalizeMessage($message);

            Mail::mailer($mailer)->send('emails.notification', [
                'title' => $subject,
                'content' => $content,
                'user' => $user,
                'data' => $data,
            ], function ($mail) use ($user, $subject) {
                $mail->to($user->email, $user->name)->subject($subject);
            });

            Log::info('Email sent', [
                'user' => $user->email,
                'mailer' => $mailer,
            ]);

            return ['success' => true, 'message' => 'Email sent successfully'];
        } catch (\Exception $e) {
            Log::error('Email failed', [
                'error' => $e->getMessage(),
                'user' => $user->email,
            ]);

            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    public function sendPushNotification(User $user, string $title, string $body, array $data = [])
    {
        try {
            if (!$this->settings->push_enabled) {
                Log::warning('Push notification skipped because channel is disabled', [
                    'user' => $user->id,
                ]);

                return [
                    'success' => false,
                    'message' => 'Push notifications are currently disabled.',
                ];
            }

            if (empty($user->device_token)) {
                Log::warning('Push notification skipped due to missing device token', [
                    'user' => $user->id,
                ]);

                return [
                    'success' => false,
                    'message' => 'Device token not available for recipient.',
                ];
            }

            $provider = $this->settings->push_provider;
            $bodyText = $this->normalizeMessage($body);
            $payload = is_array($data) ? $data : [];

            if ($provider === 'firebase') {
                return $this->sendFirebasePush($user->device_token, $title, $bodyText, $payload);
            } elseif ($provider === 'onesignal') {
                return $this->sendOneSignalPush($user->device_token, $title, $bodyText, $payload);
            }
            return ['success' => false, 'message' => 'No push provider configured'];
        } catch (\Exception $e) {
            Log::error('Push notification failed', ['error' => $e->getMessage()]);
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    protected function applyMailConfig(): string
    {
        $mailer = $this->settings->mail_mailer ?: config('mail.default');

        config(['mail.default' => $mailer]);

        if ($mailer === 'smtp') {
            $smtpConfig = [];

            if (!empty($this->settings->mail_host)) {
                $smtpConfig['mail.mailers.smtp.host'] = $this->settings->mail_host;
            }

            if (!empty($this->settings->mail_port)) {
                $smtpConfig['mail.mailers.smtp.port'] = (int) $this->settings->mail_port;
            }

            if (!empty($this->settings->mail_username)) {
                $smtpConfig['mail.mailers.smtp.username'] = $this->settings->mail_username;
            }

            if (!empty($this->settings->mail_password)) {
                $smtpConfig['mail.mailers.smtp.password'] = $this->settings->mail_password;
            }

            if (!empty($this->settings->mail_encryption)) {
                $smtpConfig['mail.mailers.smtp.encryption'] = $this->settings->mail_encryption;
            }

            if (!empty($smtpConfig)) {
                config($smtpConfig);
            }
        }

        $fromConfig = [];

        if (!empty($this->settings->mail_from_address)) {
            $fromConfig['mail.from.address'] = $this->settings->mail_from_address;
        }

        if (!empty($this->settings->mail_from_name)) {
            $fromConfig['mail.from.name'] = $this->settings->mail_from_name;
        }

        if (!empty($fromConfig)) {
            config($fromConfig);
        }

        return $mailer;
    }

    protected function normalizeMessage(mixed $message): string
    {
        if (is_string($message)) {
            return $message;
        }

        if (is_array($message)) {
            $encoded = json_encode($message, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

            return $encoded !== false ? $encoded : 'Notification details attached.';
        }

        if ($message instanceof \Stringable) {
            return (string) $message;
        }

        if (is_object($message) && method_exists($message, '__toString')) {
            return (string) $message;
        }

        return (string) $message;
    }

    protected function sendFirebasePush(string $deviceToken, string $title, string $body, array $data = [])
    {
        if (empty($this->settings->push_server_key)) {
            Log::error('Firebase push configuration missing server key.');

            return [
                'success' => false,
                'message' => 'Firebase server key is missing. Please update push settings.',
            ];
        }

        if (empty($deviceToken)) {
            Log::error('Firebase push missing device token.');

            return [
                'success' => false,
                'message' => 'Device token is required for Firebase notifications.',
            ];
        }

        $serverKey = $this->settings->push_server_key;
        $response = Http::withHeaders([
            'Authorization' => 'key=' . $serverKey,
            'Content-Type' => 'application/json',
        ])->post('https://fcm.googleapis.com/fcm/send', [
            'to' => $deviceToken,
            'notification' => ['title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => 1],
            'data' => $data,
        ]);
        if ($response->successful()) {
            Log::info('Firebase push sent');
            return ['success' => true, 'message' => 'Push notification sent'];
        }
        Log::error('Firebase push API error', [
            'status' => $response->status(),
            'body' => $response->body(),
        ]);
        return ['success' => false, 'message' => 'Failed to send push notification'];
    }

    protected function sendOneSignalPush(string $deviceToken, string $title, string $body, array $data = [])
    {
        $appId = $this->settings->push_config['app_id'] ?? '';
        $apiKey = $this->settings->push_api_key;
        if (empty($appId) || empty($apiKey)) {
            Log::error('OneSignal push configuration incomplete.', [
                'has_app_id' => !empty($appId),
                'has_api_key' => !empty($apiKey),
            ]);

            return [
                'success' => false,
                'message' => 'OneSignal configuration is incomplete. Please provide App ID and API key.',
            ];
        }

        if (empty($deviceToken)) {
            Log::error('OneSignal push missing device token.');

            return [
                'success' => false,
                'message' => 'Device token is required for OneSignal notifications.',
            ];
        }

        $response = Http::withHeaders([
            'Authorization' => 'Basic ' . $apiKey,
            'Content-Type' => 'application/json',
        ])->post('https://onesignal.com/api/v1/notifications', [
            'app_id' => $appId,
            'include_player_ids' => [$deviceToken],
            'headings' => ['en' => $title],
            'contents' => ['en' => $body],
            'data' => $data,
        ]);
        if ($response->successful()) {
            Log::info('OneSignal push sent');
            return ['success' => true, 'message' => 'Push notification sent'];
        }
        Log::error('OneSignal push API error', [
            'status' => $response->status(),
            'body' => $response->body(),
        ]);
        return ['success' => false, 'message' => 'Failed to send push notification'];
    }

    public function sendSms(User $user, string $title, string $message)
    {
        try {
            if (!$this->settings->sms_enabled) {
                Log::warning('SMS send skipped because notifications are disabled', [
                    'user' => $user->id,
                ]);

                return [
                    'success' => false,
                    'message' => 'SMS notifications are currently disabled.',
                ];
            }

            if (empty($user->phone)) {
                Log::warning('SMS send failed: recipient has no phone number', [
                    'user' => $user->id,
                ]);

                return [
                    'success' => false,
                    'message' => 'Recipient phone number is missing.',
                ];
            }

            $provider = $this->settings->sms_provider;
            $content = $this->normalizeMessage($message);

            return match ($provider) {
                'twilio' => $this->sendTwilioSms($user->phone, $content),
                'nexmo' => $this->sendNexmoSms($user->phone, $content),
                'africas_talking' => $this->sendAfricasTalkingSms($user->phone, $content),
                default => ['success' => false, 'message' => 'No SMS provider configured'],
            };
        } catch (\Exception $e) {
            Log::error('SMS send failed', ['error' => $e->getMessage()]);
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    protected function sendTwilioSms(string $phone, string $message)
    {
        if (empty($this->settings->sms_api_key) || empty($this->settings->sms_api_secret) || empty($this->settings->sms_from_number)) {
            Log::error('Twilio SMS missing configuration.');

            return [
                'success' => false,
                'message' => 'Twilio credentials are incomplete. Please update the SMS settings and try again.',
            ];
        }

        $accountSid = $this->settings->sms_api_key;
        $authToken = $this->settings->sms_api_secret;
        $fromNumber = $this->settings->sms_from_number;
        $response = Http::withBasicAuth($accountSid, $authToken)->asForm()
            ->post("https://api.twilio.com/2010-04-01/Accounts/{$accountSid}/Messages.json", [
                'To' => $phone, 'From' => $fromNumber, 'Body' => $message,
            ]);
        if ($response->successful()) {
            Log::info('Twilio SMS sent');
            return ['success' => true, 'message' => 'SMS sent successfully'];
        }
        Log::error('Twilio SMS API error', [
            'status' => $response->status(),
            'body' => $response->body(),
        ]);
        return ['success' => false, 'message' => 'Failed to send SMS'];
    }

    protected function sendNexmoSms(string $phone, string $message)
    {
        if (empty($this->settings->sms_api_key) || empty($this->settings->sms_api_secret) || empty($this->settings->sms_from_number)) {
            Log::error('Nexmo SMS missing configuration.');

            return [
                'success' => false,
                'message' => 'Nexmo credentials are incomplete. Please update the SMS settings and try again.',
            ];
        }

        $apiKey = $this->settings->sms_api_key;
        $apiSecret = $this->settings->sms_api_secret;
        $fromNumber = $this->settings->sms_from_number;
        $response = Http::post('https://rest.nexmo.com/sms/json', [
            'api_key' => $apiKey, 'api_secret' => $apiSecret,
            'from' => $fromNumber, 'to' => $phone, 'text' => $message,
        ]);
        if ($response->successful()) {
            Log::info('Nexmo SMS sent');
            return ['success' => true, 'message' => 'SMS sent successfully'];
        }
        Log::error('Nexmo SMS API error', [
            'status' => $response->status(),
            'body' => $response->body(),
        ]);
        return ['success' => false, 'message' => 'Failed to send SMS'];
    }

    protected function sendAfricasTalkingSms(string $phone, string $message)
    {
        if (empty($this->settings->sms_api_key) || empty($this->settings->sms_from_number)) {
            Log::error('Africa\'s Talking SMS missing configuration.');

            return [
                'success' => false,
                'message' => "Africa's Talking credentials are incomplete. Please update the SMS settings and try again.",
            ];
        }

        $apiKey = $this->settings->sms_api_key;
        $username = $this->settings->sms_config['username'] ?? '';
        $from = $this->settings->sms_from_number;

        if (empty($username)) {
            Log::error('Africa\'s Talking username missing.');

            return [
                'success' => false,
                'message' => "Africa's Talking username is missing. Please update the SMS settings and try again.",
            ];
        }

        $response = Http::withHeaders(['apiKey' => $apiKey, 'Content-Type' => 'application/x-www-form-urlencoded'])
            ->asForm()->post('https://api.africastalking.com/version1/messaging', [
                'username' => $username, 'to' => $phone, 'message' => $message, 'from' => $from,
            ]);
        if ($response->successful()) {
            Log::info('Africa Talking SMS sent');
            return ['success' => true, 'message' => 'SMS sent successfully'];
        }
        Log::error("Africa's Talking SMS API error", [
            'status' => $response->status(),
            'body' => $response->body(),
        ]);
        return ['success' => false, 'message' => 'Failed to send SMS'];
    }
}
