<?php

namespace App\Services;

use App\Models\Topic;
use App\Models\Lesson;
use App\Models\SiteSetting;
use Illuminate\Support\Str;

class TopicSplitterService
{
    /**
     * Check if topic content should be split based on word count
     * 
     * @param string $content
     * @param int $chunkWords
     * @return bool
     */
    public function shouldSplit($content, $chunkWords)
    {
        if (empty($content) || $chunkWords <= 0) {
            return false;
        }

        // Check if content has complex HTML formatting - don't auto-split rich formatted content
        if ($this->hasComplexHtml($content)) {
            return false;
        }

        $wordCount = str_word_count(strip_tags($content));
        
        // Split if content exceeds chunk_words threshold
        return $wordCount > $chunkWords;
    }

    /**
     * Check if content contains complex HTML formatting
     * Skip splitting only for content with rich formatting (tables, divs, complex structure)
     * Allow splitting for simple paragraph tags from TinyMCE
     * 
     * @param string $content
     * @return bool
     */
    protected function hasComplexHtml($content)
    {
        // Check for complex HTML tags that indicate rich formatting
        $complexTags = ['<table', '<div', '<img', '<iframe', '<video', '<ul', '<ol', '<h1', '<h2', '<h3', '<h4', '<h5', '<h6'];
        
        foreach ($complexTags as $tag) {
            if (stripos($content, $tag) !== false) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Check if content contains HTML tags (legacy - kept for compatibility)
     * 
     * @param string $content
     * @return bool
     */
    protected function hasHtmlTags($content)
    {
        return preg_match('/<[a-z][\s\S]*>/i', $content) === 1;
    }

    /**
     * Split topic content into multiple topics
     * 
     * @param int $lessonId
     * @param string $baseTitle
     * @param string $content
     * @param int $chunkWords
     * @param array $otherData (video_url, image_url, is_published, etc.)
     * @return array Array of created topic IDs
     */
    public function splitAndCreateTopics($lessonId, $baseTitle, $content, $chunkWords, $otherData = [])
    {
        $lesson = Lesson::find($lessonId);
        if (!$lesson) {
            throw new \Exception('Lesson not found');
        }

        // Get course chunk_words if not provided
        if (!$chunkWords || $chunkWords <= 0) {
            $chunkWords = $lesson->course->chunk_words ?? SiteSetting::getChunkWordsDefault();
        }

        \Log::info("TopicSplitter: Using chunk_words = {$chunkWords} for lesson {$lessonId}");
        \Log::info("TopicSplitter: Content word count = " . str_word_count(strip_tags($content)));

        // Check if splitting is needed
        if (!$this->shouldSplit($content, $chunkWords)) {
            // Create single topic
            \Log::info("TopicSplitter: Content not long enough to split");
            $topic = $this->createTopic($lessonId, $baseTitle, $content, $otherData);
            return [$topic->id];
        }

        // Split content into chunks
        $chunks = $this->splitContent($content, $chunkWords);
        \Log::info("TopicSplitter: Split into " . count($chunks) . " chunks");
        
        foreach ($chunks as $i => $chunk) {
            $wc = str_word_count($chunk);
            \Log::info("TopicSplitter: Chunk " . ($i+1) . " has {$wc} words");
        }
        
        $topicIds = [];

        // Get the next available order_index
        $maxOrderIndex = Topic::where('lesson_id', $lessonId)->max('order_index') ?? 0;

        // Create a topic for each chunk
        foreach ($chunks as $index => $chunk) {
            $partNumber = $index + 1;
            $totalParts = count($chunks);
            
            // Title format: "Original Title (Part 1 of 3)"
            $title = $totalParts > 1 
                ? "{$baseTitle} (Part {$partNumber} of {$totalParts})"
                : $baseTitle;

            $topicData = array_merge($otherData, [
                'order_index' => $maxOrderIndex + $index + 1
            ]);

            $topic = $this->createTopic($lessonId, $title, $chunk, $topicData);
            $topicIds[] = $topic->id;
        }

        \Log::info("TopicSplitter: Created " . count($topicIds) . " topics");
        return $topicIds;
    }

    /**
     * Split content into word-based chunks
     * 
     * @param string $content
     * @param int $chunkWords
     * @return array
     */
    protected function splitContent($content, $chunkWords)
    {
        // Strip HTML tags to get plain text
        $plainText = strip_tags($content);
        
        // Remove excessive whitespace
        $plainText = preg_replace('/\s+/', ' ', trim($plainText));
        
        // Split by words
        $words = explode(' ', $plainText);
        $chunks = [];

        for ($i = 0; $i < count($words); $i += $chunkWords) {
            $chunk = array_slice($words, $i, $chunkWords);
            $chunkText = implode(' ', $chunk);
            
            // Wrap in paragraph tag if original content had HTML
            if ($this->hasHtmlTags($content)) {
                $chunkText = '<p>' . $chunkText . '</p>';
            }
            
            $chunks[] = $chunkText;
        }

        return $chunks;
    }

    /**
     * Create a single topic
     * 
     * @param int $lessonId
     * @param string $title
     * @param string $body
     * @param array $data
     * @return Topic
     */
    protected function createTopic($lessonId, $title, $body, $data = [])
    {
        $topicData = array_merge([
            'lesson_id' => $lessonId,
            'title' => $title,
            'body' => $body,
            'is_published' => $data['is_published'] ?? true,
            'video_url' => $data['video_url'] ?? null,
            'image_url' => $data['image_url'] ?? null,
            'order_index' => $data['order_index'] ?? 0,
        ], $data);

        return Topic::create($topicData);
    }

    /**
     * Split an existing topic into multiple topics
     * 
     * @param Topic $topic
     * @param int|null $chunkWords
     * @return array Array of created topic IDs (including original)
     */
    public function splitExistingTopic(Topic $topic, $chunkWords = null)
    {
        $lesson = $topic->lesson;
        if (!$lesson || !$lesson->course) {
            throw new \Exception('Topic lesson or course not found');
        }

        if (!$chunkWords) {
            $chunkWords = $lesson->course->chunk_words ?? SiteSetting::getChunkWordsDefault();
        }

        // Check if splitting is needed
        if (!$this->shouldSplit($topic->body, $chunkWords)) {
            return [$topic->id];
        }

        // Store original data
        $originalTitle = $topic->title;
        $originalBody = $topic->body;
        $originalOrderIndex = $topic->order_index;

        // Split content
        $chunks = $this->splitContent($originalBody, $chunkWords);
        
        if (count($chunks) <= 1) {
            return [$topic->id];
        }

        $topicIds = [];

        // Update first chunk in the existing topic
        $topic->update([
            'title' => "{$originalTitle} (Part 1 of " . count($chunks) . ")",
            'body' => $chunks[0]
        ]);
        $topicIds[] = $topic->id;

        // Create new topics for remaining chunks
        for ($i = 1; $i < count($chunks); $i++) {
            $partNumber = $i + 1;
            $newTopic = Topic::create([
                'lesson_id' => $topic->lesson_id,
                'title' => "{$originalTitle} (Part {$partNumber} of " . count($chunks) . ")",
                'body' => $chunks[$i],
                'video_url' => $topic->video_url,
                'image_url' => $topic->image_url,
                'is_published' => $topic->is_published,
                'order_index' => $originalOrderIndex + $i
            ]);
            $topicIds[] = $newTopic->id;
        }

        // Reorder subsequent topics
        Topic::where('lesson_id', $topic->lesson_id)
            ->where('order_index', '>', $originalOrderIndex)
            ->where('id', '!=', $topic->id)
            ->whereNotIn('id', $topicIds)
            ->increment('order_index', count($chunks) - 1);

        return $topicIds;
    }

    /**
     * Get word count for content
     * 
     * @param string $content
     * @return int
     */
    public function getWordCount($content)
    {
        return str_word_count(strip_tags($content));
    }

    /**
     * Calculate how many topics will be created
     * 
     * @param string $content
     * @param int $chunkWords
     * @return int
     */
    public function calculateTopicCount($content, $chunkWords)
    {
        if (!$this->shouldSplit($content, $chunkWords)) {
            return 1;
        }

        $wordCount = $this->getWordCount($content);
        return (int) ceil($wordCount / $chunkWords);
    }
}
