<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Penalty rules table - configurable by admin/tutor
        Schema::create('penalty_rules', function (Blueprint $table) {
            $table->id();
            $table->string('name'); // e.g., "3-Day Inactivity Yellow Card"
            $table->text('description')->nullable();
            $table->enum('card_type', ['yellow', 'red']); // Card to issue
            $table->enum('trigger_type', ['inactivity_days', 'missed_deadline', 'failed_quiz', 'custom']); // What triggers the penalty
            $table->integer('threshold_value'); // e.g., 3 for "3 days"
            $table->integer('points_deduction'); // Points to deduct
            $table->integer('escalation_days')->nullable(); // Days before escalation (yellow to red)
            $table->integer('escalation_points')->nullable(); // Additional points to deduct on escalation
            $table->boolean('is_active')->default(true);
            $table->integer('order')->default(0); // For sorting
            $table->json('metadata')->nullable(); // Additional configuration
            $table->timestamps();
        });

        // User penalty cards table - tracks issued cards
        Schema::create('user_penalty_cards', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->foreignId('penalty_rule_id')->nullable()->constrained('penalty_rules')->onDelete('set null');
            $table->foreignId('issued_by')->nullable()->constrained('users')->onDelete('set null'); // Admin/tutor who issued it
            $table->enum('card_type', ['yellow', 'red']);
            $table->string('reason'); // Why the card was issued
            $table->integer('points_deducted')->default(0);
            $table->timestamp('issued_at');
            $table->timestamp('expires_at')->nullable(); // Optional expiry
            $table->timestamp('escalated_at')->nullable(); // When yellow became red
            $table->boolean('is_active')->default(true); // Can be revoked
            $table->timestamp('revoked_at')->nullable();
            $table->foreignId('revoked_by')->nullable()->constrained('users')->onDelete('set null');
            $table->text('revoke_reason')->nullable();
            $table->json('metadata')->nullable(); // Track related data (course_id, days_inactive, etc.)
            $table->timestamps();
            
            $table->index(['user_id', 'card_type', 'is_active']);
            $table->index('issued_at');
        });

        // Penalty history log - immutable audit trail
        Schema::create('penalty_history', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->foreignId('penalty_card_id')->nullable()->constrained('user_penalty_cards')->onDelete('set null');
            $table->enum('action_type', ['issued', 'escalated', 'revoked', 'points_deducted', 'notification_sent']);
            $table->enum('card_type', ['yellow', 'red'])->nullable();
            $table->integer('points_affected')->default(0);
            $table->text('description');
            $table->foreignId('action_by')->nullable()->constrained('users')->onDelete('set null'); // Who performed the action
            $table->json('metadata')->nullable();
            $table->timestamp('created_at');
            
            $table->index(['user_id', 'created_at']);
            $table->index('action_type');
        });

        // User activity tracking - for monitoring inactivity
        Schema::create('user_activity_log', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->enum('activity_type', ['login', 'course_access', 'lesson_view', 'quiz_attempt', 'topic_complete']);
            $table->foreignId('related_id')->nullable(); // course_id, lesson_id, etc.
            $table->string('related_type')->nullable(); // course, lesson, quiz
            $table->timestamp('activity_at');
            $table->json('metadata')->nullable();
            
            $table->index(['user_id', 'activity_at']);
            $table->index('activity_type');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('user_activity_log');
        Schema::dropIfExists('penalty_history');
        Schema::dropIfExists('user_penalty_cards');
        Schema::dropIfExists('penalty_rules');
    }
};
