import { useForm } from '@inertiajs/react';
import InputError from '@/Components/InputError';
import InputLabel from '@/Components/InputLabel';
import PrimaryButton from '@/Components/PrimaryButton';

interface QuizQuestionFormProps {
    lessonId: number;
    question?: {
        id: number;
        question: string;
        options: string[];
        correct_answer: number;
        explanation: string;
    };
    onSuccess?: () => void;
}

export default function QuizQuestionForm({ lessonId, question, onSuccess }: QuizQuestionFormProps) {
    const isEditing = !!question;

    const { data, setData, post, put, processing, errors } = useForm({
        lesson_id: lessonId,
        question: question?.question || '',
        options: question?.options || ['', '', '', ''],
        correct_answer: question?.correct_answer || 0,
        explanation: question?.explanation || '',
        order_index: 0,
        is_active: true
    });

    const submit = (e: React.FormEvent) => {
        e.preventDefault();

        if (isEditing) {
            put(route('quiz-questions.update', question.id), {
                onSuccess: onSuccess,
            });
        } else {
            post(route('quiz-questions.store'), {
                onSuccess: onSuccess,
            });
        }
    };

    const updateOption = (index: number, value: string) => {
        const newOptions = [...data.options];
        newOptions[index] = value;
        setData('options', newOptions);
    };

    return (
        <form onSubmit={submit} className="space-y-6">
            <div>
                <InputLabel htmlFor="question" value="Question" />
                <textarea
                    id="question"
                    className="mt-1 block w-full border-gray-300 focus:border-indigo-500 focus:ring-indigo-500 rounded-md shadow-sm"
                    rows={3}
                    value={data.question}
                    onChange={(e) => setData('question', e.target.value)}
                    placeholder="Enter the quiz question..."
                    required
                />
                <InputError message={errors.question} className="mt-2" />
            </div>

            <div className="space-y-4">
                <InputLabel value="Answer Options" />
                {data.options.map((option, index) => (
                    <div key={index} className="flex items-center space-x-2">
                        <span className="text-sm font-medium text-gray-700 w-8">
                            {String.fromCharCode(65 + index)}.
                        </span>
                        <input
                            type="text"
                            className="flex-1 border-gray-300 focus:border-indigo-500 focus:ring-indigo-500 rounded-md shadow-sm"
                            value={option}
                            onChange={(e) => updateOption(index, e.target.value)}
                            placeholder={`Enter option ${String.fromCharCode(65 + index)}`}
                            required
                        />
                    </div>
                ))}
                <InputError message={errors.options} className="mt-2" />
            </div>

            <div>
                <InputLabel htmlFor="correct_answer" value="Correct Answer" />
                <select
                    id="correct_answer"
                    className="mt-1 block w-full border-gray-300 focus:border-indigo-500 focus:ring-indigo-500 rounded-md shadow-sm"
                    value={data.correct_answer}
                    onChange={(e) => setData('correct_answer', parseInt(e.target.value))}
                    required
                >
                    {data.options.map((option, index) => (
                        <option key={index} value={index}>
                            {String.fromCharCode(65 + index)} - {option || `Option ${String.fromCharCode(65 + index)}`}
                        </option>
                    ))}
                </select>
                <InputError message={errors.correct_answer} className="mt-2" />
            </div>

            <div>
                <InputLabel htmlFor="explanation" value="Explanation (Optional)" />
                <textarea
                    id="explanation"
                    className="mt-1 block w-full border-gray-300 focus:border-indigo-500 focus:ring-indigo-500 rounded-md shadow-sm"
                    rows={2}
                    value={data.explanation}
                    onChange={(e) => setData('explanation', e.target.value)}
                    placeholder="Explain why this answer is correct..."
                />
                <InputError message={errors.explanation} className="mt-2" />
            </div>

            <div className="flex items-center justify-end">
                <PrimaryButton disabled={processing}>
                    {isEditing ? 'Update Question' : 'Create Question'}
                </PrimaryButton>
            </div>
        </form>
    );
}