import { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import TopicForm from '@/Components/TopicForm';
import QuizQuestionForm from '@/Components/QuizQuestionForm';

interface Topic {
    id: number;
    title: string;
    content: string;
    order_index: number;
    is_published: boolean;
}

interface QuizQuestion {
    id: number;
    lesson_id: number;
    question: string;
    options: string[];
    correct_answer: number;
    explanation: string;
    order_index: number;
    is_active: boolean;
}

interface Lesson {
    id: number;
    name: string;
    description: string;
    order_index: number;
    is_active: boolean;
    topics: Topic[];
    quiz_questions?: QuizQuestion[];
}

interface Course {
    id: number;
    name: string;
}

interface Props {
    lesson: Lesson;
    course: Course;
}

export default function LessonDetail({ lesson, course }: Props) {
    const [showTopicForm, setShowTopicForm] = useState(false);
    const [editingTopic, setEditingTopic] = useState<Topic | null>(null);
    const [showQuizForm, setShowQuizForm] = useState(false);
    const [editingQuizQuestion, setEditingQuizQuestion] = useState<QuizQuestion | null>(null);

    const handleTopicSuccess = () => {
        setShowTopicForm(false);
        setEditingTopic(null);
        router.visit(window.location.href);
    };

    const handleDeleteTopic = (topicId: number) => {
        if (confirm('Are you sure you want to delete this topic?')) {
            router.delete(route('topics.destroy', topicId), {
                onSuccess: handleTopicSuccess,
            });
        }
    };

    const handleDeleteQuizQuestion = (questionId: number) => {
        if (confirm('Are you sure you want to delete this question?')) {
            router.delete(route('quiz-questions.destroy', questionId), {
                onSuccess: () => {
                    setShowQuizForm(false);
                    setEditingQuizQuestion(null);
                    router.visit(window.location.href);
                },
            });
        }
    };

    return (
        <>
            <Head title={`${lesson.name} - Lesson Details`} />

            <div className="min-h-screen bg-gray-50">
                <div className="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
                    {/* Header */}
                    <div className="mb-8">
                        <div className="flex items-center justify-between">
                            <div>
                                <Link
                                    href={`/courses/${course.id}`}
                                    className="text-indigo-600 hover:text-indigo-800 text-sm"
                                >
                                    ← Back to {course.name}
                                </Link>
                                <h1 className="text-3xl font-bold text-gray-900 mt-2">{lesson.name}</h1>
                                <p className="mt-2 text-gray-600">{lesson.description}</p>
                            </div>
                            <div className="flex items-center space-x-4">
                                <span className={`px-3 py-1 text-sm rounded-full ${
                                    lesson.is_active
                                        ? 'bg-green-100 text-green-800'
                                        : 'bg-yellow-100 text-yellow-800'
                                }`}>
                                    {lesson.is_active ? 'Active' : 'Inactive'}
                                </span>
                                <span className="text-sm text-gray-500">
                                    Order: #{lesson.order_index}
                                </span>
                                <Link
                                    href={`/lessons/${lesson.id}/edit`}
                                    className="bg-indigo-600 text-white px-4 py-2 rounded-md hover:bg-indigo-700 text-sm"
                                >
                                    Edit Lesson
                                </Link>
                            </div>
                        </div>
                    </div>

                    {/* Topics Section */}
                    <div className="bg-white shadow rounded-lg">
                        <div className="px-6 py-4 border-b border-gray-200">
                            <div className="flex justify-between items-center">
                                <h2 className="text-lg font-medium text-gray-900">Topics</h2>
                                <button
                                    onClick={() => setShowTopicForm(true)}
                                    className="bg-indigo-600 text-white px-4 py-2 rounded-md hover:bg-indigo-700 text-sm"
                                >
                                    Add Topic
                                </button>
                            </div>
                        </div>

                        <div className="p-6">
                            {showTopicForm && (
                                <div className="mb-6 border rounded-lg p-4 bg-gray-50">
                                    <h3 className="text-lg font-medium mb-4">
                                        {editingTopic ? 'Edit Topic' : 'Create New Topic'}
                                    </h3>
                                    <TopicForm
                                        lessonId={lesson.id}
                                        topic={editingTopic || undefined}
                                        onSuccess={handleTopicSuccess}
                                    />
                                    <button
                                        onClick={() => {
                                            setShowTopicForm(false);
                                            setEditingTopic(null);
                                        }}
                                        className="mt-4 text-gray-600 hover:text-gray-800 text-sm"
                                    >
                                        Cancel
                                    </button>
                                </div>
                            )}

                            {lesson.topics && lesson.topics.length > 0 ? (
                                <div className="space-y-4">
                                    {lesson.topics
                                        .sort((a, b) => a.order_index - b.order_index)
                                        .map((topic) => (
                                            <div key={topic.id} className="border rounded-lg p-4">
                                                <div className="flex justify-between items-start">
                                                    <div className="flex-1">
                                                        <div className="flex items-center space-x-3">
                                                            <span className="text-sm text-gray-500 font-medium">
                                                                #{topic.order_index}
                                                            </span>
                                                            <h3 className="text-lg font-medium text-gray-900">
                                                                {topic.title}
                                                            </h3>
                                                            <span className={`px-2 py-1 text-xs rounded-full ${
                                                                topic.is_published
                                                                    ? 'bg-green-100 text-green-800'
                                                                    : 'bg-yellow-100 text-yellow-800'
                                                            }`}>
                                                                {topic.is_published ? 'Published' : 'Draft'}
                                                            </span>
                                                        </div>
                                                        <div className="mt-3 text-gray-600 line-clamp-3">
                                                            {topic.content.substring(0, 200)}
                                                            {topic.content.length > 200 && '...'}
                                                        </div>
                                                    </div>
                                                    <div className="flex space-x-2 ml-4">
                                                        <button
                                                            onClick={() => {
                                                                setEditingTopic(topic);
                                                                setShowTopicForm(true);
                                                            }}
                                                            className="text-indigo-600 hover:text-indigo-800 text-sm"
                                                        >
                                                            Edit
                                                        </button>
                                                        <button
                                                            onClick={() => handleDeleteTopic(topic.id)}
                                                            className="text-red-600 hover:text-red-800 text-sm"
                                                        >
                                                            Delete
                                                        </button>
                                                        <Link
                                                            href={`/topics/${topic.id}`}
                                                            className="text-gray-600 hover:text-gray-800 text-sm"
                                                        >
                                                            View
                                                        </Link>
                                                    </div>
                                                </div>
                                            </div>
                                        ))}
                                </div>
                            ) : (
                                <div className="text-center py-8">
                                    <p className="text-gray-600 mb-4">No topics created yet.</p>
                                    <button
                                        onClick={() => setShowTopicForm(true)}
                                        className="bg-indigo-600 text-white px-4 py-2 rounded-md hover:bg-indigo-700"
                                    >
                                        Create Your First Topic
                                    </button>
                                </div>
                            )}
                        </div>
                    </div>

                    {/* Quiz Questions Section */}
                    <div className="bg-white shadow rounded-lg mt-8">
                        <div className="px-6 py-4 border-b border-gray-200">
                            <div className="flex justify-between items-center">
                                <h2 className="text-lg font-medium text-gray-900">Quiz Questions</h2>
                                <button
                                    onClick={() => setShowQuizForm(true)}
                                    className="bg-indigo-600 text-white px-4 py-2 rounded-md hover:bg-indigo-700 text-sm"
                                >
                                    Add Question
                                </button>
                            </div>
                        </div>

                        <div className="p-6">
                            {showQuizForm && (
                                <div className="mb-6 border rounded-lg p-4 bg-gray-50">
                                    <h3 className="text-lg font-medium mb-4">
                                        {editingQuizQuestion ? 'Edit Quiz Question' : 'Create New Quiz Question'}
                                    </h3>
                                    <QuizQuestionForm
                                        lessonId={lesson.id}
                                        question={editingQuizQuestion || undefined}
                                        onSuccess={() => {
                                            setShowQuizForm(false);
                                            setEditingQuizQuestion(null);
                                            router.visit(window.location.href);
                                        }}
                                    />
                                    <button
                                        onClick={() => {
                                            setShowQuizForm(false);
                                            setEditingQuizQuestion(null);
                                        }}
                                        className="mt-4 text-gray-600 hover:text-gray-800 text-sm"
                                    >
                                        Cancel
                                    </button>
                                </div>
                            )}

                            {lesson.quiz_questions && lesson.quiz_questions.length > 0 ? (
                                <div className="space-y-4">
                                    {lesson.quiz_questions
                                        .sort((a, b) => a.order_index - b.order_index)
                                        .map((question) => (
                                            <div key={question.id} className="border rounded-lg p-4">
                                                <div className="flex justify-between items-start">
                                                    <div className="flex-1">
                                                        <div className="flex items-center space-x-3">
                                                            <span className="text-sm text-gray-500 font-medium">
                                                                Q{question.order_index}
                                                            </span>
                                                            <span className={`px-2 py-1 text-xs rounded-full ${
                                                                question.is_active
                                                                    ? 'bg-green-100 text-green-800'
                                                                    : 'bg-yellow-100 text-yellow-800'
                                                            }`}>
                                                                {question.is_active ? 'Active' : 'Inactive'}
                                                            </span>
                                                        </div>
                                                        <div className="mt-2">
                                                            <p className="font-medium text-gray-900">{question.question}</p>
                                                            <div className="mt-2 space-y-1">
                                                                {question.options.map((option, index) => (
                                                                    <div key={index} className={`text-sm p-2 rounded ${
                                                                        index === question.correct_answer
                                                                            ? 'bg-green-50 border border-green-200 text-green-800'
                                                                            : 'bg-gray-50 text-gray-600'
                                                                    }`}>
                                                                        {index + 1}. {option}
                                                                        {index === question.correct_answer && (
                                                                            <span className="ml-2 text-green-600 font-medium">(Correct)</span>
                                                                        )}
                                                                    </div>
                                                                ))}
                                                            </div>
                                                            {question.explanation && (
                                                                <div className="mt-3 p-3 bg-blue-50 border border-blue-200 rounded">
                                                                    <p className="text-sm text-blue-800">
                                                                        <strong>Explanation:</strong> {question.explanation}
                                                                    </p>
                                                                </div>
                                                            )}
                                                        </div>
                                                    </div>
                                                    <div className="flex space-x-2 ml-4">
                                                        <button
                                                            onClick={() => {
                                                                setEditingQuizQuestion(question);
                                                                setShowQuizForm(true);
                                                            }}
                                                            className="text-indigo-600 hover:text-indigo-800 text-sm"
                                                        >
                                                            Edit
                                                        </button>
                                                        <button
                                                            onClick={() => handleDeleteQuizQuestion(question.id)}
                                                            className="text-red-600 hover:text-red-800 text-sm"
                                                        >
                                                            Delete
                                                        </button>
                                                    </div>
                                                </div>
                                            </div>
                                        ))}
                                </div>
                            ) : (
                                <div className="text-center py-8">
                                    <p className="text-gray-600 mb-4">No quiz questions created yet.</p>
                                    <button
                                        onClick={() => setShowQuizForm(true)}
                                        className="bg-indigo-600 text-white px-4 py-2 rounded-md hover:bg-indigo-700"
                                    >
                                        Create Your First Question
                                    </button>
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </>
    );
}