<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>Certificate Template Builder - Admin</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f7fa;
            padding: 20px;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
        }

        .header {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            margin-bottom: 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .header h1 {
            color: #2c3e50;
            font-size: 2em;
        }

        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            transition: all 0.3s;
        }

        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(102, 126, 234, 0.4);
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-danger {
            background: #dc3545;
            color: white;
        }

        .btn-success {
            background: #28a745;
            color: white;
        }

        .templates-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .template-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            transition: transform 0.2s;
        }

        .template-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }

        .template-preview {
            width: 100%;
            height: 200px;
            background: #f8f9fa;
            border-radius: 8px;
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            border: 2px dashed #dee2e6;
            position: relative;
            overflow: hidden;
        }

        .template-preview img {
            max-width: 100%;
            max-height: 100%;
            object-fit: cover;
        }

        .template-info h3 {
            color: #2c3e50;
            margin-bottom: 10px;
        }

        .template-info p {
            color: #6c757d;
            font-size: 14px;
            margin-bottom: 15px;
        }

        .template-actions {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }

        .template-actions .btn {
            flex: 1;
            min-width: 80px;
            padding: 8px 12px;
            font-size: 12px;
        }

        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 1000;
            overflow-y: auto;
        }

        .modal.active {
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }

        .modal-content {
            background: white;
            border-radius: 10px;
            padding: 30px;
            max-width: 900px;
            width: 100%;
            max-height: 90vh;
            overflow-y: auto;
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f0f0f0;
        }

        .modal-header h2 {
            color: #2c3e50;
        }

        .close-btn {
            font-size: 28px;
            color: #6c757d;
            cursor: pointer;
            background: none;
            border: none;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            margin-bottom: 8px;
            color: #2c3e50;
            font-weight: 600;
        }

        .form-group input,
        .form-group textarea,
        .form-group select {
            width: 100%;
            padding: 12px;
            border: 2px solid #e1e5e9;
            border-radius: 6px;
            font-size: 14px;
        }

        .form-group textarea {
            resize: vertical;
            min-height: 100px;
        }

        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }

        .certificate-builder {
            margin-top: 30px;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 8px;
        }

        .certificate-canvas {
            width: 100%;
            height: 500px;
            background: white;
            border: 2px solid #dee2e6;
            border-radius: 8px;
            position: relative;
            overflow: hidden;
            margin-bottom: 20px;
        }

        .preview-text {
            white-space: nowrap;
            user-select: none;
            transition: all 0.2s ease;
        }

        .draggable {
            cursor: move;
            pointer-events: auto;
        }

        .qr-placeholder {
            position: absolute;
            border: 2px dashed #6c757d;
            color: #6c757d;
            background: rgba(108, 117, 125, 0.05);
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            letter-spacing: 1px;
            user-select: none;
            pointer-events: auto;
        }

        .selected {
            outline: 2px dashed #667eea;
            outline-offset: 2px;
        }

        .text-field-config {
            background: white;
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 10px;
            border: 1px solid #dee2e6;
        }

        .text-field-config h4 {
            margin-bottom: 10px;
            color: #495057;
        }

        .field-controls {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 10px;
        }

        .field-controls input,
        .field-controls select {
            padding: 8px;
            border: 1px solid #ced4da;
            border-radius: 4px;
            font-size: 12px;
        }

        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #6c757d;
        }

        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            opacity: 0.5;
        }

        .badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }

        .badge-success {
            background: #d4edda;
            color: #155724;
        }

        .badge-secondary {
            background: #e2e3e5;
            color: #383d41;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 6px;
            margin-bottom: 20px;
        }

        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div>
                <h1>🎓 Certificate Template Builder</h1>
                <p style="color: #6c757d; margin-top: 8px;">Create and manage certificate templates for courses and lessons</p>
            </div>
            <button class="btn btn-primary" onclick="openCreateModal()">
                + Create New Template
            </button>
        </div>

        <div id="alertContainer"></div>

        <div class="templates-grid" id="templatesGrid">
            <div class="empty-state">
                <div style="font-size: 64px; margin-bottom: 20px;">📜</div>
                <h3>No certificate templates yet</h3>
                <p>Create your first certificate template to get started</p>
            </div>
        </div>
    </div>

    <!-- Create/Edit Template Modal -->
    <div class="modal" id="templateModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="modalTitle">Create Certificate Template</h2>
                <button class="close-btn" onclick="closeModal()">&times;</button>
            </div>

            <form id="templateForm" onsubmit="saveTemplate(event)">
                <input type="hidden" id="templateId">

                <div class="form-row">
                    <div class="form-group">
                        <label>Template Name *</label>
                        <input type="text" id="templateName" required placeholder="e.g., Course Completion Certificate">
                    </div>
                    <div class="form-group">
                        <label>Orientation *</label>
                        <select id="orientation" required>
                            <option value="landscape">Landscape</option>
                            <option value="portrait">Portrait</option>
                        </select>
                    </div>
                </div>

                <div class="form-group">
                    <label>Description</label>
                    <textarea id="description" placeholder="Brief description of this template..."></textarea>
                </div>

                <div class="form-group">
                    <label>Background Image</label>
                    <input type="file" id="backgroundImage" accept="image/*">
                    <small style="color: #6c757d;">Upload a background image for the certificate (optional)</small>
                </div>

                <div class="certificate-builder">
                    <h3 style="margin-bottom: 8px;">Certificate Layout</h3>
                    <p style="margin-bottom: 12px; color:#6b7280; font-size: 13px;">
                        Tip: Click and drag the labels (Name, Course, Date, Code) and the QR box directly on the certificate to position them. You can also use your keyboard arrows (Shift for bigger steps). Use the inputs below to fine‑tune.
                    </p>
                    
                    <div class="certificate-canvas" id="certificateCanvas">
                        <div id="previewBackground" style="width: 100%; height: 100%; position: relative; background: white;">
                            <!-- Preview text elements will be dynamically inserted here -->
                            <div id="previewName" class="preview-text draggable" style="position: absolute; font-family: Arial; font-size: 32px; color: #000000; text-align: center; left: 50%; transform: translateX(-50%); top: 200px;">
                                John Doe
                            </div>
                            <div id="previewCourse" class="preview-text draggable" style="position: absolute; font-family: Arial; font-size: 24px; color: #333333; text-align: center; left: 50%; transform: translateX(-50%); top: 280px;">
                                Advanced Programming Course
                            </div>
                            <div id="previewDate" class="preview-text draggable" style="position: absolute; font-family: Arial; font-size: 18px; color: #666666; text-align: center; left: 50%; transform: translateX(-50%); top: 360px;">
                                October 18, 2025
                            </div>
                            <div id="previewCode" class="preview-text draggable" style="position: absolute; font-family: 'Courier New'; font-size: 14px; color: #999999; text-align: center; left: 50%; transform: translateX(-50%); top: 440px;">
                                CERT-ABC123XYZ
                            </div>
                            <div id="previewQR" class="qr-placeholder" style="left: 85%; top: 650px; width: 100px; height: 100px; transform: translateX(-50%);">QR</div>
                        </div>
                    </div>

                    <h4 style="margin-bottom: 15px;">Text Fields Configuration</h4>
                    
                    <div class="text-field-config">
                        <h4 style="display:flex; align-items:center; justify-content:space-between;">
                            <span>Student Name</span>
                            <button type="button" class="btn btn-secondary" style="padding:6px 10px; font-size:12px;" onclick="resetField('name')">Reset</button>
                        </h4>
                        <div class="field-controls">
                            <input type="number" placeholder="X Position" id="name_x" value="50">
                            <input type="number" placeholder="Y Position" id="name_y" value="200">
                            <input type="number" placeholder="Font Size" id="name_size" value="32">
                            <input type="text" placeholder="Font Family" id="name_font" value="Arial">
                            <input type="color" id="name_color" value="#000000">
                            <select id="name_align">
                                <option value="left">Left</option>
                                <option value="center" selected>Center</option>
                                <option value="right">Right</option>
                            </select>
                        </div>
                    </div>

                    <div class="text-field-config">
                        <h4 style="display:flex; align-items:center; justify-content:space-between;">
                            <span>Course/Lesson Title</span>
                            <button type="button" class="btn btn-secondary" style="padding:6px 10px; font-size:12px;" onclick="resetField('course')">Reset</button>
                        </h4>
                        <div class="field-controls">
                            <input type="number" placeholder="X Position" id="course_x" value="50">
                            <input type="number" placeholder="Y Position" id="course_y" value="280">
                            <input type="number" placeholder="Font Size" id="course_size" value="24">
                            <input type="text" placeholder="Font Family" id="course_font" value="Arial">
                            <input type="color" id="course_color" value="#333333">
                            <select id="course_align">
                                <option value="left">Left</option>
                                <option value="center" selected>Center</option>
                                <option value="right">Right</option>
                            </select>
                        </div>
                    </div>

                    <div class="text-field-config">
                        <h4 style="display:flex; align-items:center; justify-content:space-between;">
                            <span>Completion Date</span>
                            <button type="button" class="btn btn-secondary" style="padding:6px 10px; font-size:12px;" onclick="resetField('date')">Reset</button>
                        </h4>
                        <div class="field-controls">
                            <input type="number" placeholder="X Position" id="date_x" value="50">
                            <input type="number" placeholder="Y Position" id="date_y" value="360">
                            <input type="number" placeholder="Font Size" id="date_size" value="18">
                            <input type="text" placeholder="Font Family" id="date_font" value="Arial">
                            <input type="color" id="date_color" value="#666666">
                            <select id="date_align">
                                <option value="left">Left</option>
                                <option value="center" selected>Center</option>
                                <option value="right">Right</option>
                            </select>
                        </div>
                    </div>

                    <div class="text-field-config">
                        <h4 style="display:flex; align-items:center; justify-content:space-between;">
                            <span>Certificate Code</span>
                            <button type="button" class="btn btn-secondary" style="padding:6px 10px; font-size:12px;" onclick="resetField('code')">Reset</button>
                        </h4>
                        <div class="field-controls">
                            <input type="number" placeholder="X Position" id="code_x" value="50">
                            <input type="number" placeholder="Y Position" id="code_y" value="440">
                            <input type="number" placeholder="Font Size" id="code_size" value="14">
                            <input type="text" placeholder="Font Family" id="code_font" value="Courier New">
                            <input type="color" id="code_color" value="#999999">
                            <select id="code_align">
                                <option value="left">Left</option>
                                <option value="center" selected>Center</option>
                                <option value="right">Right</option>
                            </select>
                        </div>
                    </div>

                    <h4 style="margin: 30px 0 15px 0; border-top: 2px solid #e1e5e9; padding-top: 20px;">QR Code Configuration</h4>
                    
                    <div class="text-field-config">
                        <div style="display: flex; align-items: center; margin-bottom: 15px;">
                            <input type="checkbox" id="qr_code_enabled" checked style="width: auto; margin-right: 10px;">
                            <label for="qr_code_enabled" style="margin: 0; cursor: pointer;">Enable QR Code for certificate verification</label>
                        </div>
                        
                        <div id="qrCodeConfig">
                            <h4 style="font-size: 14px; margin-bottom: 10px; color: #6c757d; display:flex; align-items:center; justify-content:space-between;">
                                <span>QR Code Position & Size</span>
                                <button type="button" class="btn btn-secondary" style="padding:6px 10px; font-size:12px;" onclick="resetQR()">Reset</button>
                            </h4>
                            <div class="field-controls">
                                <input type="number" placeholder="X Position %" id="qr_x" value="85" min="0" max="100">
                                <input type="number" placeholder="Y Position (px)" id="qr_y" value="650" min="0">
                                <input type="number" placeholder="Size (px)" id="qr_size" value="100" min="50" max="300">
                            </div>
                            <small style="color: #6c757d; display: block; margin-top: 8px;">
                                💡 Tip: Position the QR code in bottom-right corner for best results
                            </small>
                        </div>
                    </div>
                </div>

                <div style="display: flex; gap: 10px; margin-top: 30px;">
                    <button type="submit" class="btn btn-primary" style="flex: 1;">Save Template</button>
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
                </div>
            </form>
        </div>
    </div>

    <script>
    let templates = [];
    let editingTemplateId = null;
    let API_BASE = '/admin/certificates';
    let selectedElementKey = null; // 'name' | 'course' | 'date' | 'code' | 'qr'
    const DEFAULTS = {
        name: { x: 50, y: 200 },
        course: { x: 50, y: 280 },
        date: { x: 50, y: 360 },
        code: { x: 50, y: 440 },
        qr: { x: 85, y: 650, size: 100 }
    };

        // Load templates on page load
        document.addEventListener('DOMContentLoaded', function() {
            // Determine API base for admin or tutor context
            const path = window.location.pathname;
            API_BASE = path.startsWith('/admin') ? '/admin/certificates' : '/tutor/certificates';

            // Initial load
            loadTemplates();

            // Setup drag behavior for preview elements
            setupDraggables();

            // Hook up QR toggle
            const qrEnabled = document.getElementById('qr_code_enabled');
            const qrConfig = document.getElementById('qrCodeConfig');
            qrEnabled.addEventListener('change', () => {
                qrConfig.style.display = qrEnabled.checked ? 'block' : 'none';
                // Always show QR in builder for design; dim if disabled
                const previewQR = document.getElementById('previewQR');
                previewQR.style.display = 'flex';
                previewQR.style.opacity = qrEnabled.checked ? '1' : '0.5';
            });
            // Initialize visibility
            qrConfig.style.display = qrEnabled.checked ? 'block' : 'none';
            document.getElementById('previewQR').style.display = 'flex';
            document.getElementById('previewQR').style.opacity = qrEnabled.checked ? '1' : '0.5';

            // Selection handlers for keyboard nudging
            const clickableMap = [
                { key: 'name', id: 'previewName' },
                { key: 'course', id: 'previewCourse' },
                { key: 'date', id: 'previewDate' },
                { key: 'code', id: 'previewCode' },
                { key: 'qr', id: 'previewQR' },
            ];
            clickableMap.forEach(({key, id}) => {
                const el = document.getElementById(id);
                el.addEventListener('click', (e) => {
                    e.stopPropagation();
                    selectElement(key);
                });
            });

            // Deselect on background click
            document.getElementById('previewBackground').addEventListener('click', () => selectElement(null));

            // Keyboard arrow support
            document.addEventListener('keydown', (e) => {
                const modalOpen = document.getElementById('templateModal').classList.contains('active');
                if (!modalOpen || !selectedElementKey) return;
                if (!["ArrowUp","ArrowDown","ArrowLeft","ArrowRight"].includes(e.key)) return;
                e.preventDefault();
                const container = document.getElementById('previewBackground');
                const rect = container.getBoundingClientRect();
                const isQR = selectedElementKey === 'qr';
                const stepY = e.shiftKey ? 10 : 1;
                const stepX = e.shiftKey ? 5 : 1; // percent

                const xInputId = isQR ? 'qr_x' : `${selectedElementKey}_x`;
                const yInputId = isQR ? 'qr_y' : `${selectedElementKey}_y`;
                let x = parseInt(document.getElementById(xInputId).value || '0');
                let y = parseInt(document.getElementById(yInputId).value || '0');

                if (e.key === 'ArrowLeft') x = Math.max(0, x - stepX);
                if (e.key === 'ArrowRight') x = Math.min(100, x + stepX);
                if (e.key === 'ArrowUp') y = Math.max(0, y - stepY);
                if (e.key === 'ArrowDown') y = Math.min(Math.round(rect.height), y + stepY);

                document.getElementById(xInputId).value = x;
                document.getElementById(yInputId).value = y;
                if (isQR) updateQRPreview(); else updatePreview(selectedElementKey);
            });
        });

        async function loadTemplates() {
            try {
                const response = await fetch(API_BASE);
                const data = await response.json();
                
                if (data.success) {
                    templates = data.templates;
                    renderTemplates();
                }
            } catch (error) {
                console.error('Failed to load templates:', error);
                showAlert('Failed to load templates', 'error');
            }
        }

        function renderTemplates() {
            const grid = document.getElementById('templatesGrid');
            
            if (templates.length === 0) {
                grid.innerHTML = `
                    <div class="empty-state">
                        <div style="font-size: 64px; margin-bottom: 20px;">📜</div>
                        <h3>No certificate templates yet</h3>
                        <p>Create your first certificate template to get started</p>
                    </div>
                `;
                return;
            }

            grid.innerHTML = templates.map(template => `
                <div class="template-card">
                    <div class="template-preview">
                        ${template.background_image 
                            ? `<img src="/storage/${template.background_image}" alt="${template.name}">`
                            : `<div style="color: #6c757d;">
                                <div style="font-size: 48px; margin-bottom: 10px;">📜</div>
                                <p>No background image</p>
                               </div>`
                        }
                    </div>
                    <div class="template-info">
                        <h3>${template.name}</h3>
                        <p>${template.description || 'No description'}</p>
                        <div style="margin-bottom: 10px;">
                            <span class="badge ${template.is_active ? 'badge-success' : 'badge-secondary'}">
                                ${template.is_active ? 'Active' : 'Inactive'}
                            </span>
                            <span class="badge badge-secondary">${template.orientation}</span>
                        </div>
                        <div class="template-actions">
                            <button class="btn btn-primary" onclick="editTemplate(${template.id})">Edit</button>
                            <button class="btn btn-secondary" onclick="duplicateTemplate(${template.id})">Duplicate</button>
                            <button class="btn btn-success" onclick="openAssignModal(${template.id})">Assign</button>
                            <button class="btn" onclick="previewTemplate(${template.id})">Preview</button>
                            <button class="btn btn-danger" onclick="deleteTemplate(${template.id})">Delete</button>
                        </div>
                    </div>
                </div>
            `).join('');
        }

        function openCreateModal() {
            editingTemplateId = null;
            document.getElementById('modalTitle').textContent = 'Create Certificate Template';
            document.getElementById('templateForm').reset();
            
            // Reset canvas and preview
            const canvas = document.getElementById('certificateCanvas');
            canvas.style.height = '500px';
            
            const previewBg = document.getElementById('previewBackground');
            previewBg.style.backgroundImage = 'none';
            previewBg.style.background = 'white';
            
            // Reset all preview fields to default values
            const fields = ['name', 'course', 'date', 'code'];
            fields.forEach(field => updatePreview(field));

            // Reset QR preview and inputs to defaults
            document.getElementById('qr_code_enabled').checked = true;
            document.getElementById('qr_x').value = 85;
            document.getElementById('qr_y').value = 650;
            document.getElementById('qr_size').value = 100;
            document.getElementById('qrCodeConfig').style.display = 'block';
            document.getElementById('previewQR').style.display = 'flex';
            updateQRPreview();
            
            document.getElementById('templateModal').classList.add('active');
        }

        function closeModal() {
            document.getElementById('templateModal').classList.remove('active');
        }

        async function editTemplate(id) {
            try {
                const response = await fetch(`${API_BASE}/${id}`);
                const data = await response.json();
                
                if (data.success) {
                    const template = data.template;
                    editingTemplateId = id;
                    
                    document.getElementById('modalTitle').textContent = 'Edit Certificate Template';
                    document.getElementById('templateName').value = template.name;
                    document.getElementById('description').value = template.description || '';
                    document.getElementById('orientation').value = template.orientation;
                    
                    // Update canvas height based on orientation
                    const canvas = document.getElementById('certificateCanvas');
                    canvas.style.height = template.orientation === 'portrait' ? '700px' : '500px';
                    
                    // Load background image if exists
                    const previewBg = document.getElementById('previewBackground');
                    if (template.background_image) {
                        previewBg.style.backgroundImage = `url(/storage/${template.background_image})`;
                        previewBg.style.backgroundSize = 'cover';
                        previewBg.style.backgroundPosition = 'center';
                    } else {
                        previewBg.style.backgroundImage = 'none';
                        previewBg.style.background = 'white';
                    }
                    
                    // Populate text fields
                    if (template.text_fields) {
                        for (const [key, field] of Object.entries(template.text_fields)) {
                            document.getElementById(`${key}_x`).value = field.x || 50;
                            document.getElementById(`${key}_y`).value = field.y || 200;
                            document.getElementById(`${key}_size`).value = field.size || 24;
                            document.getElementById(`${key}_font`).value = field.font || 'Arial';
                            document.getElementById(`${key}_color`).value = field.color || '#000000';
                            document.getElementById(`${key}_align`).value = field.align || 'center';
                            
                            // Update preview for each field
                            updatePreview(key);
                        }
                    }

                    // Populate QR settings
                    const qrEnabled = document.getElementById('qr_code_enabled');
                    const qrConfig = document.getElementById('qrCodeConfig');
                    const previewQR = document.getElementById('previewQR');
                    if (typeof template.qr_code_enabled !== 'undefined') {
                        qrEnabled.checked = !!template.qr_code_enabled;
                    }
                    if (template.qr_code_config) {
                        document.getElementById('qr_x').value = template.qr_code_config.x ?? 85;
                        document.getElementById('qr_y').value = template.qr_code_config.y ?? 650;
                        document.getElementById('qr_size').value = template.qr_code_config.size ?? 100;
                    }
                    qrConfig.style.display = qrEnabled.checked ? 'block' : 'none';
                    // Always show QR for design; dim if disabled
                    previewQR.style.display = 'flex';
                    previewQR.style.opacity = qrEnabled.checked ? '1' : '0.5';
                    updateQRPreview();
                    
                    document.getElementById('templateModal').classList.add('active');
                }
            } catch (error) {
                console.error('Failed to load template:', error);
                showAlert('Failed to load template', 'error');
            }
        }

        async function saveTemplate(event) {
            event.preventDefault();
            
            const formData = new FormData();
            formData.append('name', document.getElementById('templateName').value);
            formData.append('description', document.getElementById('description').value);
            formData.append('orientation', document.getElementById('orientation').value);

            // layout_config[] as array (Laravel validation expects array)
            formData.append('layout_config[version]', '1');

            // text_fields[] as array brackets
            const fields = ['name', 'course', 'date', 'code'];
            fields.forEach((key) => {
                formData.append(`text_fields[${key}][x]`, parseInt(document.getElementById(`${key}_x`).value));
                formData.append(`text_fields[${key}][y]`, parseInt(document.getElementById(`${key}_y`).value));
                formData.append(`text_fields[${key}][size]`, parseInt(document.getElementById(`${key}_size`).value));
                formData.append(`text_fields[${key}][font]`, document.getElementById(`${key}_font`).value);
                formData.append(`text_fields[${key}][color]`, document.getElementById(`${key}_color`).value);
                formData.append(`text_fields[${key}][align]`, document.getElementById(`${key}_align`).value);
            });
            
            // Add QR code configuration
            const qrCodeEnabled = document.getElementById('qr_code_enabled').checked;
            formData.append('qr_code_enabled', qrCodeEnabled ? '1' : '0');
            if (qrCodeEnabled) {
                formData.append('qr_code_config[x]', parseInt(document.getElementById('qr_x').value));
                formData.append('qr_code_config[y]', parseInt(document.getElementById('qr_y').value));
                formData.append('qr_code_config[size]', parseInt(document.getElementById('qr_size').value));
            }
            
            const backgroundImage = document.getElementById('backgroundImage').files[0];
            if (backgroundImage) {
                formData.append('background_image', backgroundImage);
            }
            
            try {
                const url = editingTemplateId 
                    ? `${API_BASE}/${editingTemplateId}`
                    : `${API_BASE}`;
                    
                const method = editingTemplateId ? 'PUT' : 'POST';
                
                // For PUT requests with FormData, we need to use POST with _method override
                if (editingTemplateId) {
                    formData.append('_method', 'PUT');
                }
                
                const response = await fetch(url, {
                    method: 'POST',
                    body: formData,
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                    }
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showAlert(data.message, 'success');
                    closeModal();
                    loadTemplates();
                } else {
                    showAlert(data.message || 'Failed to save template', 'error');
                }
            } catch (error) {
                console.error('Failed to save template:', error);
                showAlert('Failed to save template', 'error');
            }
        }

        async function duplicateTemplate(id) {
            if (!confirm('Duplicate this certificate template?')) return;
            
            try {
                const response = await fetch(`${API_BASE}/${id}/duplicate`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                    }
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showAlert(data.message, 'success');
                    loadTemplates();
                } else {
                    showAlert(data.message || 'Failed to duplicate template', 'error');
                }
            } catch (error) {
                console.error('Failed to duplicate template:', error);
                showAlert('Failed to duplicate template', 'error');
            }
        }

        async function deleteTemplate(id) {
            if (!confirm('Are you sure you want to delete this template?')) return;
            
            try {
                const response = await fetch(`${API_BASE}/${id}`, {
                    method: 'DELETE',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                    }
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showAlert(data.message, 'success');
                    loadTemplates();
                } else {
                    showAlert(data.message || 'Failed to delete template', 'error');
                }
            } catch (error) {
                console.error('Failed to delete template:', error);
                showAlert('Failed to delete template', 'error');
            }
        }

        function previewTemplate(id) {
            const url = `${API_BASE}/${id}/preview`;
            window.open(url, '_blank');
        }

        function showAlert(message, type = 'success') {
            const alertContainer = document.getElementById('alertContainer');
            const alertClass = type === 'success' ? 'alert-success' : 'alert-error';
            
            const alert = document.createElement('div');
            alert.className = `alert ${alertClass}`;
            alert.textContent = message;
            
            alertContainer.appendChild(alert);
            
            setTimeout(() => {
                alert.remove();
            }, 5000);
        }

        // Live preview update functions
        function updatePreview(field) {
            const previewElement = document.getElementById(`preview${field.charAt(0).toUpperCase() + field.slice(1)}`);
            if (!previewElement) return;

            const x = document.getElementById(`${field}_x`).value;
            const y = document.getElementById(`${field}_y`).value;
            const size = document.getElementById(`${field}_size`).value;
            const font = document.getElementById(`${field}_font`).value;
            const color = document.getElementById(`${field}_color`).value;
            const align = document.getElementById(`${field}_align`).value;

            // Update position (convert percentage to actual position)
            previewElement.style.left = x + '%';
            previewElement.style.top = y + 'px';
            
            // Update font properties
            previewElement.style.fontSize = size + 'px';
            previewElement.style.fontFamily = font;
            previewElement.style.color = color;
            previewElement.style.textAlign = align;
            
            // Adjust transform based on alignment
            if (align === 'center') {
                previewElement.style.transform = 'translateX(-50%)';
            } else if (align === 'right') {
                previewElement.style.transform = 'translateX(-100%)';
            } else {
                previewElement.style.transform = 'translateX(0)';
            }
        }

        function updateQRPreview() {
            const qr = document.getElementById('previewQR');
            const x = parseInt(document.getElementById('qr_x').value || '85');
            const y = parseInt(document.getElementById('qr_y').value || '650');
            const size = parseInt(document.getElementById('qr_size').value || '100');
            qr.style.left = x + '%';
            qr.style.top = y + 'px';
            qr.style.width = size + 'px';
            qr.style.height = size + 'px';
            qr.style.transform = 'translateX(-50%)';
            const enabled = document.getElementById('qr_code_enabled').checked;
            qr.style.opacity = enabled ? '1' : '0.5';
        }

        function selectElement(key) {
            selectedElementKey = key;
            const ids = {
                name: 'previewName',
                course: 'previewCourse',
                date: 'previewDate',
                code: 'previewCode',
                qr: 'previewQR'
            };
            // Clear previous selections
            Object.values(ids).forEach(id => {
                const el = document.getElementById(id);
                if (el) el.classList.remove('selected');
            });
            if (key && ids[key]) {
                const el = document.getElementById(ids[key]);
                if (el) el.classList.add('selected');
            }
        }

        function resetField(field) {
            const def = DEFAULTS[field];
            if (!def) return;
            document.getElementById(`${field}_x`).value = def.x;
            document.getElementById(`${field}_y`).value = def.y;
            updatePreview(field);
            selectElement(field);
        }

        function resetQR() {
            const def = DEFAULTS.qr;
            document.getElementById('qr_x').value = def.x;
            document.getElementById('qr_y').value = def.y;
            document.getElementById('qr_size').value = def.size;
            updateQRPreview();
            selectElement('qr');
        }

        // Add event listeners for all text field inputs
        document.addEventListener('DOMContentLoaded', function() {
            const fields = ['name', 'course', 'date', 'code'];
            const properties = ['x', 'y', 'size', 'font', 'color', 'align'];

            fields.forEach(field => {
                properties.forEach(prop => {
                    const element = document.getElementById(`${field}_${prop}`);
                    if (element) {
                        element.addEventListener('input', () => updatePreview(field));
                        element.addEventListener('change', () => updatePreview(field));
                    }
                });
            });

            // Handle background image preview
            document.getElementById('backgroundImage').addEventListener('change', function(e) {
                if (e.target.files && e.target.files[0]) {
                    const reader = new FileReader();
                    reader.onload = function(event) {
                        document.getElementById('previewBackground').style.backgroundImage = `url(${event.target.result})`;
                        document.getElementById('previewBackground').style.backgroundSize = 'cover';
                        document.getElementById('previewBackground').style.backgroundPosition = 'center';
                    };
                    reader.readAsDataURL(e.target.files[0]);
                }
            });

            // Handle orientation change
            document.getElementById('orientation').addEventListener('change', function() {
                const canvas = document.getElementById('certificateCanvas');
                if (this.value === 'portrait') {
                    canvas.style.height = '700px';
                } else {
                    canvas.style.height = '500px';
                }
            });

            // QR input listeners
            ['qr_x','qr_y','qr_size'].forEach(id => {
                const el = document.getElementById(id);
                el.addEventListener('input', updateQRPreview);
                el.addEventListener('change', updateQRPreview);
            });
        });

        function setupDraggables() {
            const container = document.getElementById('previewBackground');
            const map = [
                { id: 'previewName', x: 'name_x', y: 'name_y' },
                { id: 'previewCourse', x: 'course_x', y: 'course_y' },
                { id: 'previewDate', x: 'date_x', y: 'date_y' },
                { id: 'previewCode', x: 'code_x', y: 'code_y' }
            ];

            map.forEach(({id, x, y}) => makeDraggable(document.getElementById(id), container, x, y, (field) => updatePreview(field)));
            // QR draggable
            makeDraggable(document.getElementById('previewQR'), container, 'qr_x', 'qr_y', () => updateQRPreview(), true);
        }

        function makeDraggable(el, container, xInputId, yInputId, onChange, isQR = false) {
            if (!el || !container) return;
            let dragging = false;

            const onMouseDown = (e) => {
                dragging = true;
                e.preventDefault();
                // Select this element for keyboard control
                const field = isQR ? 'qr' : xInputId.split('_')[0];
                selectElement(field);
            };
            const onMouseUp = () => dragging = false;
            const onMouseMove = (e) => {
                if (!dragging) return;
                const rect = container.getBoundingClientRect();
                const relX = Math.min(Math.max(e.clientX - rect.left, 0), rect.width);
                const relY = Math.min(Math.max(e.clientY - rect.top, 0), rect.height);
                const percentX = (relX / rect.width) * 100;

                const xInput = document.getElementById(xInputId);
                const yInput = document.getElementById(yInputId);
                xInput.value = Math.round(percentX);
                yInput.value = Math.round(relY);

                if (isQR) {
                    updateQRPreview();
                } else {
                    const field = xInputId.split('_')[0];
                    updatePreview(field);
                }
            };

            el.addEventListener('mousedown', onMouseDown);
            document.addEventListener('mouseup', onMouseUp);
            document.addEventListener('mousemove', onMouseMove);
            // Touch support for mobile
            const onTouchStart = (e) => { dragging = true; e.preventDefault(); };
            const onTouchEnd = () => { dragging = false; };
            const onTouchMove = (e) => {
                if (!dragging) return;
                const t = e.touches[0];
                const rect = container.getBoundingClientRect();
                const relX = Math.min(Math.max(t.clientX - rect.left, 0), rect.width);
                const relY = Math.min(Math.max(t.clientY - rect.top, 0), rect.height);
                const percentX = (relX / rect.width) * 100;
                const xInput = document.getElementById(xInputId);
                const yInput = document.getElementById(yInputId);
                xInput.value = Math.round(percentX);
                yInput.value = Math.round(relY);
                if (isQR) { updateQRPreview(); } else { const field = xInputId.split('_')[0]; updatePreview(field); }
            };
            el.addEventListener('touchstart', onTouchStart, { passive: false });
            document.addEventListener('touchend', onTouchEnd, { passive: false });
            document.addEventListener('touchmove', onTouchMove, { passive: false });
        }
    </script>

    <!-- Assignment Modal -->
    <div class="modal" id="assignModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Assign Certificate Template</h2>
                <button class="close-btn" onclick="closeAssignModal()">&times;</button>
            </div>
            <div class="form-group">
                <label>Assign To</label>
                <select id="assignType">
                    <option value="course">Course</option>
                    <option value="lesson">Lesson</option>
                </select>
            </div>
            <div class="form-group" id="assignCourseGroup">
                <label>Select Course</label>
                <select id="assignCourse"></select>
            </div>
            <div class="form-group" id="assignLessonGroup" style="display:none;">
                <label>Select Lesson</label>
                <select id="assignLesson"></select>
            </div>
            <div style="display:flex; gap:10px;">
                <button class="btn btn-primary" onclick="submitAssignment()">Assign</button>
                <button class="btn btn-secondary" onclick="closeAssignModal()">Cancel</button>
            </div>
        </div>
    </div>

    <script>
        let assigningTemplateId = null;

        async function openAssignModal(templateId) {
            assigningTemplateId = templateId;
            document.getElementById('assignModal').classList.add('active');
            document.getElementById('assignType').value = 'course';
            document.getElementById('assignLessonGroup').style.display = 'none';
            await loadCourses();
        }

        function closeAssignModal() {
            document.getElementById('assignModal').classList.remove('active');
        }

        document.getElementById('assignType').addEventListener('change', async function() {
            const type = this.value;
            if (type === 'course') {
                document.getElementById('assignLessonGroup').style.display = 'none';
                await loadCourses();
            } else {
                document.getElementById('assignLessonGroup').style.display = 'block';
                await loadCourses(true);
            }
        });

        async function loadCourses(forLessons = false) {
            try {
                const base = API_BASE.startsWith('/admin') ? '/admin' : '/tutor';
                const res = await fetch(`${base}/courses`);
                const data = await res.json();
                const courses = Array.isArray(data) ? data : (data.courses || data.data || data);

                const select = document.getElementById('assignCourse');
                select.innerHTML = '';
                (courses || []).forEach(c => {
                    const opt = document.createElement('option');
                    opt.value = c.id;
                    opt.textContent = c.title || c.name || `Course #${c.id}`;
                    select.appendChild(opt);
                });

                if (forLessons) {
                    await loadLessonsForSelectedCourse();
                    select.onchange = loadLessonsForSelectedCourse;
                }
            } catch (e) {
                console.error('Failed to load courses', e);
            }
        }

        async function loadLessonsForSelectedCourse() {
            try {
                const base = API_BASE.startsWith('/admin') ? '/admin' : '/tutor';
                const courseId = document.getElementById('assignCourse').value;
                const res = await fetch(`${base}/courses/${courseId}/lessons`);
                const data = await res.json();
                const lessons = Array.isArray(data) ? data : (data.lessons || data.data || data);
                const select = document.getElementById('assignLesson');
                select.innerHTML = '';
                (lessons || []).forEach(l => {
                    const opt = document.createElement('option');
                    opt.value = l.id;
                    opt.textContent = l.title || l.name || `Lesson #${l.id}`;
                    select.appendChild(opt);
                });
            } catch (e) {
                console.error('Failed to load lessons', e);
            }
        }

        async function submitAssignment() {
            try {
                const type = document.getElementById('assignType').value;
                const courseId = document.getElementById('assignCourse').value;
                const lessonId = document.getElementById('assignLesson').value;
                const assignableType = type === 'course' ? 'App\\Models\\Course' : 'App\\Models\\Lesson';
                const assignableId = type === 'course' ? courseId : lessonId;

                const form = new FormData();
                form.append('certificate_template_id', assigningTemplateId);
                form.append('assignable_type', assignableType);
                form.append('assignable_id', assignableId);

                const res = await fetch(`${API_BASE}/assign`, {
                    method: 'POST',
                    headers: { 'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content },
                    body: form
                });
                const data = await res.json();
                if (data.success) {
                    showAlert('Template assigned successfully', 'success');
                    closeAssignModal();
                } else {
                    showAlert(data.message || 'Failed to assign template', 'error');
                }
            } catch (e) {
                console.error('Assignment failed', e);
                showAlert('Assignment failed', 'error');
            }
        }
    </script>
</body>
</html>