<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>Payment Gateway Settings - Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
        }

        .header {
            background: white;
            padding: 30px;
            border-radius: 15px;
            margin-bottom: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }

        .header h1 {
            color: #333;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .header p {
            color: #666;
            font-size: 0.95em;
        }

        .back-btn {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 10px 20px;
            background: #667eea;
            color: white;
            text-decoration: none;
            border-radius: 8px;
            margin-bottom: 20px;
            transition: all 0.3s;
        }

        .back-btn:hover {
            background: #764ba2;
            transform: translateY(-2px);
        }

        .gateway-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(500px, 1fr));
            gap: 25px;
            margin-top: 30px;
        }

        .gateway-card {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
            transition: all 0.3s;
        }

        .gateway-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0,0,0,0.3);
        }

        .gateway-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
            padding-bottom: 20px;
            border-bottom: 2px solid #f0f0f0;
        }

        .gateway-title {
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .gateway-icon {
            width: 50px;
            height: 50px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 10px;
            font-size: 24px;
        }

        .gateway-icon.stripe { background: linear-gradient(135deg, #635bff 0%, #4f46e5 100%); color: white; }
        .gateway-icon.paypal { background: linear-gradient(135deg, #0070ba 0%, #003087 100%); color: white; }
        .gateway-icon.paystack { background: linear-gradient(135deg, #00c3f8 0%, #0097d6 100%); color: white; }
        .gateway-icon.crypto { background: linear-gradient(135deg, #f7931a 0%, #f2a900 100%); color: white; }

        .gateway-name h3 {
            color: #333;
            font-size: 1.5em;
        }

        .gateway-status {
            font-size: 0.85em;
            padding: 5px 15px;
            border-radius: 20px;
            font-weight: 600;
        }

        .status-enabled {
            background: #d4edda;
            color: #155724;
        }

        .status-disabled {
            background: #f8d7da;
            color: #721c24;
        }

        .toggle-switch {
            position: relative;
            width: 60px;
            height: 30px;
            background: #ccc;
            border-radius: 15px;
            cursor: pointer;
            transition: all 0.3s;
        }

        .toggle-switch.active {
            background: #4CAF50;
        }

        .toggle-switch-slider {
            position: absolute;
            top: 3px;
            left: 3px;
            width: 24px;
            height: 24px;
            background: white;
            border-radius: 50%;
            transition: all 0.3s;
        }

        .toggle-switch.active .toggle-switch-slider {
            left: 33px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            color: #333;
            font-weight: 600;
            margin-bottom: 8px;
            font-size: 0.95em;
        }

        .form-control {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 0.95em;
            transition: all 0.3s;
        }

        .form-control:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }

        .form-select {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 0.95em;
            background: white;
            cursor: pointer;
        }

        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 8px;
            font-size: 1em;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }

        .btn-success {
            background: #4CAF50;
            color: white;
        }

        .btn-danger {
            background: #f44336;
            color: white;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            display: none;
        }

        .alert-success {
            background: #d4edda;
            color: #155724;
            border-left: 4px solid #28a745;
        }

        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border-left: 4px solid #dc3545;
        }

        .mode-badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 0.8em;
            font-weight: 600;
            margin-left: 10px;
        }

        .mode-sandbox {
            background: #fff3cd;
            color: #856404;
        }

        .mode-live {
            background: #d4edda;
            color: #155724;
        }

        .webhook-url {
            background: #f8f9fa;
            padding: 12px 15px;
            border-radius: 8px;
            font-family: 'Courier New', monospace;
            font-size: 0.85em;
            color: #495057;
            word-break: break-all;
            border: 1px solid #dee2e6;
        }

        .copy-btn {
            padding: 8px 15px;
            background: #6c757d;
            color: white;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.85em;
            margin-top: 8px;
            transition: all 0.3s;
        }

        .copy-btn:hover {
            background: #5a6268;
        }

        .info-box {
            background: #e7f3ff;
            border-left: 4px solid #2196F3;
            padding: 15px;
            border-radius: 8px;
            margin-top: 15px;
        }

        .info-box h4 {
            color: #1976D2;
            margin-bottom: 8px;
            font-size: 0.95em;
        }

        .info-box p {
            color: #666;
            font-size: 0.85em;
            line-height: 1.6;
        }

        @media (max-width: 768px) {
            .gateway-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <a href="{{ route('admin.panel') }}" class="back-btn">
            <i class="fas fa-arrow-left"></i>
            Back to Dashboard
        </a>

        <div class="header">
            <h1>
                <i class="fas fa-credit-card"></i>
                Payment Gateway Settings
            </h1>
            <p>Configure and manage payment gateways for course purchases</p>
        </div>

        <div id="alertContainer"></div>

        <div class="gateway-grid">
            @foreach($settings as $key => $setting)
            <div class="gateway-card" data-gateway="{{ $key }}">
                <div class="gateway-header">
                    <div class="gateway-title">
                        <div class="gateway-icon {{ $key }}">
                            @if($key === 'stripe')
                                <i class="fab fa-stripe"></i>
                            @elseif($key === 'paypal')
                                <i class="fab fa-paypal"></i>
                            @elseif($key === 'paystack')
                                <i class="fas fa-wallet"></i>
                            @elseif($key === 'crypto')
                                <i class="fab fa-bitcoin"></i>
                            @endif
                        </div>
                        <div class="gateway-name">
                            <h3>{{ $gateways[$key] }}</h3>
                            <span class="gateway-status {{ $setting->is_enabled ? 'status-enabled' : 'status-disabled' }}">
                                {{ $setting->is_enabled ? 'Enabled' : 'Disabled' }}
                            </span>
                            <span class="mode-badge {{ $setting->mode === 'live' ? 'mode-live' : 'mode-sandbox' }}">
                                {{ strtoupper($setting->mode) }}
                            </span>
                        </div>
                    </div>
                    <div class="toggle-switch {{ $setting->is_enabled ? 'active' : '' }}" 
                         onclick="toggleGateway('{{ $key }}', this)">
                        <div class="toggle-switch-slider"></div>
                    </div>
                </div>

                <form id="form-{{ $key }}" onsubmit="saveGatewaySettings(event, '{{ $key }}')">
                    <div class="form-group">
                        <label>Mode</label>
                        <select name="mode" class="form-select" required>
                            <option value="sandbox" {{ $setting->mode === 'sandbox' ? 'selected' : '' }}>Sandbox (Test)</option>
                            <option value="live" {{ $setting->mode === 'live' ? 'selected' : '' }}>Live (Production)</option>
                        </select>
                    </div>

                    @if($key === 'stripe')
                    <div class="form-group">
                        <label>Publishable Key</label>
                        <input type="text" name="public_key" class="form-control" 
                               placeholder="pk_test_..." {{ $setting->is_enabled ? 'required' : '' }}>
                    </div>
                    <div class="form-group">
                        <label>Secret Key</label>
                        <input type="password" name="secret_key" class="form-control" 
                               placeholder="sk_test_..." {{ $setting->is_enabled ? 'required' : '' }}>
                    </div>
                    <div class="form-group">
                        <label>Webhook Secret</label>
                        <input type="password" name="webhook_secret" class="form-control" 
                               placeholder="whsec_...">
                    </div>

                    @elseif($key === 'paypal')
                    <div class="form-group">
                        <label>Client ID</label>
                        <input type="text" name="public_key" class="form-control" 
                               placeholder="Your PayPal Client ID" {{ $setting->is_enabled ? 'required' : '' }}>
                    </div>
                    <div class="form-group">
                        <label>Secret</label>
                        <input type="password" name="secret_key" class="form-control" 
                               placeholder="Your PayPal Secret" {{ $setting->is_enabled ? 'required' : '' }}>
                    </div>

                    @elseif($key === 'paystack')
                    <div class="form-group">
                        <label>Public Key</label>
                        <input type="text" name="public_key" class="form-control" 
                               placeholder="pk_test_..." {{ $setting->is_enabled ? 'required' : '' }}>
                    </div>
                    <div class="form-group">
                        <label>Secret Key</label>
                        <input type="password" name="secret_key" class="form-control" 
                               placeholder="sk_test_..." {{ $setting->is_enabled ? 'required' : '' }}>
                    </div>

                    @elseif($key === 'crypto')
                    <div class="form-group">
                        <label>API Key</label>
                        <input type="text" name="public_key" class="form-control" 
                               placeholder="Your API Key" {{ $setting->is_enabled ? 'required' : '' }}>
                    </div>
                    <div class="form-group">
                        <label>API Secret</label>
                        <input type="password" name="secret_key" class="form-control" 
                               placeholder="Your API Secret" {{ $setting->is_enabled ? 'required' : '' }}>
                    </div>
                    @endif

                    <div class="info-box">
                        <h4><i class="fas fa-link"></i> Webhook URL</h4>
                        <div class="webhook-url">{{ url('/api/webhooks/' . $key) }}</div>
                        <button type="button" class="copy-btn" onclick="copyWebhookUrl('{{ url('/api/webhooks/' . $key) }}')">
                            <i class="fas fa-copy"></i> Copy URL
                        </button>
                        <p style="margin-top: 10px;">Configure this URL in your {{ $gateways[$key] }} dashboard to receive payment notifications.</p>
                    </div>

                    <button type="submit" class="btn btn-primary" style="margin-top: 20px; width: 100%;">
                        <i class="fas fa-save"></i>
                        Save {{ $gateways[$key] }} Settings
                    </button>
                </form>
            </div>
            @endforeach
        </div>
    </div>

    <script>
        const csrfToken = document.querySelector('meta[name="csrf-token"]').content;

        function showAlert(message, type = 'success') {
            const alertContainer = document.getElementById('alertContainer');
            const alert = document.createElement('div');
            alert.className = `alert alert-${type}`;
            alert.style.display = 'block';
            alert.innerHTML = `
                <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-circle'}"></i>
                ${message}
            `;
            alertContainer.innerHTML = '';
            alertContainer.appendChild(alert);
            
            setTimeout(() => {
                alert.style.display = 'none';
            }, 5000);

            // Scroll to top
            window.scrollTo({ top: 0, behavior: 'smooth' });
        }

        async function toggleGateway(gateway, element) {
            try {
                const response = await fetch(`/admin/payment-settings/${gateway}/toggle`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': csrfToken
                    }
                });

                const data = await response.json();

                if (data.success) {
                    element.classList.toggle('active');
                    
                    const card = element.closest('.gateway-card');
                    const statusBadge = card.querySelector('.gateway-status');
                    
                    if (data.is_enabled) {
                        statusBadge.textContent = 'Enabled';
                        statusBadge.className = 'gateway-status status-enabled';
                    } else {
                        statusBadge.textContent = 'Disabled';
                        statusBadge.className = 'gateway-status status-disabled';
                    }

                    showAlert(data.message, 'success');
                } else {
                    showAlert('Failed to toggle gateway', 'error');
                }
            } catch (error) {
                console.error('Error:', error);
                showAlert('An error occurred', 'error');
            }
        }

        async function saveGatewaySettings(event, gateway) {
            event.preventDefault();
            
            const form = event.target;
            const formData = new FormData(form);
            
            const data = {
                is_enabled: form.closest('.gateway-card').querySelector('.toggle-switch').classList.contains('active'),
                mode: formData.get('mode'),
                public_key: formData.get('public_key'),
                secret_key: formData.get('secret_key'),
                webhook_secret: formData.get('webhook_secret')
            };

            try {
                const response = await fetch(`/admin/payment-settings/${gateway}`, {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': csrfToken
                    },
                    body: JSON.stringify(data)
                });

                const result = await response.json();

                if (result.success) {
                    showAlert(result.message, 'success');
                    
                    // Update mode badge
                    const card = form.closest('.gateway-card');
                    const modeBadge = card.querySelector('.mode-badge');
                    modeBadge.textContent = data.mode.toUpperCase();
                    modeBadge.className = `mode-badge mode-${data.mode}`;
                    
                    // Clear password fields for security
                    form.querySelectorAll('input[type="password"]').forEach(input => {
                        input.value = '';
                    });
                } else {
                    showAlert('Failed to save settings', 'error');
                }
            } catch (error) {
                console.error('Error:', error);
                showAlert('An error occurred while saving', 'error');
            }
        }

        function copyWebhookUrl(url) {
            navigator.clipboard.writeText(url).then(() => {
                showAlert('Webhook URL copied to clipboard!', 'success');
            });
        }
    </script>
</body>
</html>
