<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tutor API Tester</title>
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 10px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
        }
        h1 {
            color: #667eea;
            margin-bottom: 10px;
        }
        .user-info {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 30px;
        }
        .section {
            margin-bottom: 30px;
            padding-bottom: 30px;
            border-bottom: 2px solid #e9ecef;
        }
        .section:last-child {
            border-bottom: none;
        }
        h2 {
            color: #495057;
            margin-bottom: 15px;
        }
        .button-group {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-bottom: 15px;
        }
        button {
            padding: 10px 20px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            transition: all 0.3s;
        }
        .btn-primary {
            background: #667eea;
            color: white;
        }
        .btn-success {
            background: #10b981;
            color: white;
        }
        .btn-danger {
            background: #ef4444;
            color: white;
        }
        .btn-info {
            background: #3b82f6;
            color: white;
        }
        button:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }
        .result {
            background: #1e293b;
            color: #e2e8f0;
            padding: 15px;
            border-radius: 6px;
            font-family: 'Courier New', monospace;
            font-size: 13px;
            max-height: 400px;
            overflow-y: auto;
            white-space: pre-wrap;
            word-wrap: break-word;
        }
        .result.success {
            border-left: 4px solid #10b981;
        }
        .result.error {
            border-left: 4px solid #ef4444;
        }
        .input-group {
            margin-bottom: 15px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: 600;
            color: #495057;
        }
        input[type="text"],
        input[type="number"],
        textarea {
            width: 100%;
            padding: 10px;
            border: 1px solid #cbd5e1;
            border-radius: 6px;
            font-size: 14px;
        }
        textarea {
            min-height: 80px;
            font-family: inherit;
        }
        .status {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        .status.online {
            background: #d1fae5;
            color: #065f46;
        }
        .status.offline {
            background: #fee2e2;
            color: #991b1b;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🎓 Tutor API Testing Dashboard</h1>
        
        <div class="user-info">
            <strong>Logged in as:</strong> {{ Auth::user()->name }} ({{ Auth::user()->email }})
            <br>
            <strong>Role:</strong> {{ Auth::user()->getRoleNames()->first() }}
            <br>
            <strong>User ID:</strong> {{ Auth::user()->id }}
            <br>
            <strong>Server:</strong> <span class="status online">Online</span>
        </div>

        <!-- Courses Section -->
        <div class="section">
            <h2>📚 Courses Management</h2>
            <div class="button-group">
                <button class="btn-primary" onclick="getCourses()">Get All Courses</button>
                <button class="btn-success" onclick="showCreateCourseForm()">Create New Course</button>
            </div>
            
            <div id="createCourseForm" style="display: none; background: #f8f9fa; padding: 15px; border-radius: 6px; margin-bottom: 15px;">
                <h3 style="margin-bottom: 15px;">Create New Course</h3>
                <div class="input-group">
                    <label>Course Name *</label>
                    <input type="text" id="courseName" placeholder="e.g., Advanced React Development">
                </div>
                <div class="input-group">
                    <label>Short Description</label>
                    <textarea id="courseDescription" placeholder="Brief description of the course..."></textarea>
                </div>
                <div class="input-group">
                    <label>
                        <input type="checkbox" id="courseActive" checked> Active
                    </label>
                </div>
                <div class="button-group">
                    <button class="btn-success" onclick="createCourse()">Create Course</button>
                    <button class="btn-secondary" onclick="hideCreateCourseForm()">Cancel</button>
                </div>
            </div>
            
            <div id="coursesResult" class="result" style="display: none;"></div>
        </div>

        <!-- Students Section -->
        <div class="section">
            <h2>👥 Students Management</h2>
            <div class="button-group">
                <button class="btn-info" onclick="getStudents()">Get All Students</button>
            </div>
            <div id="studentsResult" class="result" style="display: none;"></div>
        </div>

        <!-- Lessons Section -->
        <div class="section">
            <h2>📖 Lessons Management</h2>
            <div class="input-group">
                <label>Course ID</label>
                <input type="number" id="lessonCourseId" placeholder="e.g., 18">
            </div>
            <div class="button-group">
                <button class="btn-primary" onclick="getLessons()">Get Lessons for Course</button>
            </div>
            <div id="lessonsResult" class="result" style="display: none;"></div>
        </div>

        <!-- Topics Section -->
        <div class="section">
            <h2>📝 Topics Management</h2>
            <div class="input-group">
                <label>Lesson ID</label>
                <input type="number" id="topicLessonId" placeholder="e.g., 1">
            </div>
            <div class="button-group">
                <button class="btn-primary" onclick="getTopics()">Get Topics for Lesson</button>
            </div>
            <div id="topicsResult" class="result" style="display: none;"></div>
        </div>

        <!-- Quiz Questions Section -->
        <div class="section">
            <h2>❓ Quiz Questions Management</h2>
            <div class="input-group">
                <label>Lesson ID</label>
                <input type="number" id="quizLessonId" placeholder="e.g., 1">
            </div>
            <div class="button-group">
                <button class="btn-primary" onclick="getQuizQuestions()">Get Quiz Questions</button>
            </div>
            <div id="quizResult" class="result" style="display: none;"></div>
        </div>
    </div>

    <script>
        // API Helper
        async function apiCall(url, method = 'GET', data = null) {
            const options = {
                method: method,
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                    'Accept': 'application/json'
                }
            };

            if (data && (method === 'POST' || method === 'PUT')) {
                options.body = JSON.stringify(data);
            }

            try {
                const response = await fetch(url, options);
                const result = await response.json();
                
                return {
                    success: response.ok,
                    status: response.status,
                    data: result
                };
            } catch (error) {
                return {
                    success: false,
                    status: 0,
                    data: { message: error.message }
                };
            }
        }

        // Display Result
        function showResult(elementId, data, success) {
            const element = document.getElementById(elementId);
            element.style.display = 'block';
            element.className = 'result ' + (success ? 'success' : 'error');
            element.textContent = JSON.stringify(data, null, 2);
        }

        // Courses
        async function getCourses() {
            const result = await apiCall('/tutor/courses');
            showResult('coursesResult', result, result.success);
        }

        function showCreateCourseForm() {
            document.getElementById('createCourseForm').style.display = 'block';
        }

        function hideCreateCourseForm() {
            document.getElementById('createCourseForm').style.display = 'none';
        }

        async function createCourse() {
            const name = document.getElementById('courseName').value;
            const description = document.getElementById('courseDescription').value;
            const isActive = document.getElementById('courseActive').checked;

            if (!name) {
                alert('Please enter a course name');
                return;
            }

            const result = await apiCall('/tutor/courses', 'POST', {
                name: name,
                short_description: description,
                is_active: isActive
            });

            showResult('coursesResult', result, result.success);
            
            if (result.success) {
                hideCreateCourseForm();
                document.getElementById('courseName').value = '';
                document.getElementById('courseDescription').value = '';
                alert('Course created successfully!');
            }
        }

        // Students
        async function getStudents() {
            const result = await apiCall('/tutor/students');
            showResult('studentsResult', result, result.success);
        }

        // Lessons
        async function getLessons() {
            const courseId = document.getElementById('lessonCourseId').value;
            if (!courseId) {
                alert('Please enter a course ID');
                return;
            }
            const result = await apiCall(`/tutor/courses/${courseId}/lessons`);
            showResult('lessonsResult', result, result.success);
        }

        // Topics
        async function getTopics() {
            const lessonId = document.getElementById('topicLessonId').value;
            if (!lessonId) {
                alert('Please enter a lesson ID');
                return;
            }
            const result = await apiCall(`/tutor/lessons/${lessonId}/topics`);
            showResult('topicsResult', result, result.success);
        }

        // Quiz Questions
        async function getQuizQuestions() {
            const lessonId = document.getElementById('quizLessonId').value;
            if (!lessonId) {
                alert('Please enter a lesson ID');
                return;
            }
            const result = await apiCall(`/tutor/lessons/${lessonId}/quiz-questions`);
            showResult('quizResult', result, result.success);
        }

        // Load courses on page load
        window.addEventListener('DOMContentLoaded', function() {
            getCourses();
        });
    </script>
</body>
</html>
