@extends('layouts.public')

@section('title', 'Training Courses - EntrepreneurEducation')
@section('meta_description', 'Browse our comprehensive training courses. Webinars, video courses, audio content and eBooks for entrepreneurs. Simple, Easy, Enjoyable & Accessible.')

@push('styles')
<style>
    :root {
        --ee-purple: #7c3aed;
        --ee-cyan: #06b6d4;
        --ee-yellow: #f59e0b;
        --ee-green: #10b981;
        --ee-dark: #0f172a;
        --ee-light: #f8fafc;
        --ee-card: #ffffff;
        --ee-border: #e2e8f0;
        --ee-text: #334155;
        --ee-text-light: #64748b;
        --ee-glow: rgba(124, 58, 237, 0.15);
    }
    
    .page-header {
        background: linear-gradient(135deg, var(--ee-purple) 0%, var(--ee-cyan) 100%);
        color: white;
        padding: 100px 0 80px;
        text-align: center;
        position: relative;
        overflow: hidden;
    }
    
    .page-header::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: 
            radial-gradient(circle at 20% 80%, rgba(255, 255, 255, 0.1) 0%, transparent 50%),
            radial-gradient(circle at 80% 20%, rgba(255, 255, 255, 0.1) 0%, transparent 50%);
    }
    
    .page-header-content {
        position: relative;
        z-index: 1;
    }
    
    .page-header h1 {
        font-size: 3.5rem;
        font-weight: 900;
        margin-bottom: 20px;
        background: linear-gradient(135deg, white, rgba(255, 255, 255, 0.9));
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        background-clip: text;
    }
    
    .page-header p {
        font-size: 1.25rem;
        opacity: 0.95;
        max-width: 700px;
        margin: 0 auto;
        color: rgba(255, 255, 255, 0.95);
    }
    
    .courses-section {
        padding: 80px 0;
        background: var(--ee-light);
    }
    
    /* Modern Search & Filters */
    .courses-filters {
        display: flex;
        gap: 20px;
        margin-bottom: 40px;
        flex-wrap: wrap;
        align-items: center;
        justify-content: space-between;
        background: white;
        padding: 24px;
        border-radius: 20px;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.05);
        border: 2px solid var(--ee-border);
    }
    
    /* Language Selector Styles */
    .language-selector {
        display: flex;
        align-items: center;
        gap: 10px;
    }
    
    .language-selector-label {
        font-size: 0.9rem;
        font-weight: 700;
        color: var(--ee-purple);
        display: flex;
        align-items: center;
        gap: 6px;
    }
    
    .language-selector select {
        padding: 10px 12px;
        border: 2px solid var(--ee-border);
        border-radius: 10px;
        background: white;
        color: var(--ee-text);
        font-weight: 600;
        min-width: 200px;
        box-shadow: 0 2px 6px rgba(0,0,0,0.05);
    }
    
    .language-selector select:focus {
        outline: none;
        border-color: var(--ee-purple);
        box-shadow: 0 0 0 4px var(--ee-glow);
    }
    
    /* Category Filter Dropdown */
    .category-filter-dropdown {
        display: flex;
        align-items: center;
        gap: 10px;
    }
    
    .category-filter-label {
        font-size: 0.9rem;
        font-weight: 700;
        color: var(--ee-purple);
        display: flex;
        align-items: center;
        gap: 6px;
    }
    
    .category-filter-dropdown select {
        padding: 10px 12px;
        border: 2px solid var(--ee-border);
        border-radius: 10px;
        background: white;
        color: var(--ee-text);
        font-weight: 600;
        min-width: 200px;
        box-shadow: 0 2px 6px rgba(0,0,0,0.05);
        cursor: pointer;
    }
    
    .category-filter-dropdown select:focus {
        outline: none;
        border-color: var(--ee-purple);
        box-shadow: 0 0 0 4px var(--ee-glow);
    }
    
    .search-box {
        flex: 1;
        min-width: 300px;
        position: relative;
    }
    
    .search-box input {
        width: 100%;
        padding: 16px 50px 16px 20px;
        border: 2px solid var(--ee-border);
        border-radius: 12px;
        font-size: 1rem;
        transition: all 0.3s ease;
        background: var(--ee-light);
        color: var(--ee-text);
        font-weight: 500;
    }
    
    .search-box input:focus {
        outline: none;
        border-color: var(--ee-purple);
        box-shadow: 0 0 0 4px var(--ee-glow);
        background: white;
    }
    
    .search-box i {
        position: absolute;
        right: 20px;
        top: 50%;
        transform: translateY(-50%);
        color: var(--ee-purple);
        font-size: 1.25rem;
    }
    
    .filter-buttons {
        display: flex;
        gap: 12px;
        flex-wrap: wrap;
    }
    
    .filter-btn {
        padding: 12px 24px;
        border: 2px solid var(--ee-border);
        background: white;
        border-radius: 12px;
        color: var(--ee-text);
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        font-size: 0.9375rem;
        position: relative;
        overflow: hidden;
    }
    
    .filter-btn:hover {
        border-color: var(--ee-purple);
        background: rgba(124, 58, 237, 0.05);
        transform: translateY(-2px);
    }
    
    .filter-btn.active {
        border-color: var(--ee-purple);
        background: linear-gradient(135deg, var(--ee-purple), var(--ee-cyan));
        color: white;
        box-shadow: 0 8px 24px var(--ee-glow);
    }
    
    .courses-count {
        color: var(--ee-purple);
        font-size: 1rem;
        margin-bottom: 20px;
        font-weight: 600;
        padding: 12px 20px;
        background: white;
        border-radius: 12px;
        display: inline-block;
        border: 2px solid var(--ee-border);
    }
    
    /* Modern Course Cards */
    .courses-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
        gap: 30px;
    }
    
    .course-card {
        background: white;
        border-radius: 24px;
        overflow: hidden;
        transition: all 0.5s cubic-bezier(0.4, 0, 0.2, 1);
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.05);
        border: 2px solid var(--ee-border);
        cursor: pointer;
        position: relative;
        transform-style: preserve-3d;
    }
    
    .course-card:hover {
        transform: translateY(-12px) scale(1.02);
        box-shadow: 0 20px 60px rgba(124, 58, 237, 0.15);
        border-color: var(--ee-purple);
    }
    
    .course-image {
        position: relative;
        height: 200px;
        overflow: hidden;
        background: linear-gradient(135deg, var(--ee-purple), var(--ee-cyan));
        display: flex;
        align-items: center;
        justify-content: center;
    }
    
    .course-image::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: linear-gradient(135deg, rgba(0,0,0,0.1), transparent);
    }
    
    .course-image img {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        object-fit: cover;
    }
    
    .course-image .course-image-placeholder {
        position: absolute;
        inset: 0;
        display: flex;
        align-items: center;
        justify-content: center;
        color: rgba(255, 255, 255, 0.2);
        font-size: 64px;
        z-index: 1;
    }
    
    /* SMALLER FREE BADGE */
    .course-badge {
        position: absolute;
        top: 15px;
        right: 15px;
        background: rgba(255, 255, 255, 0.95);
        color: var(--ee-purple);
        padding: 6px 12px;
        border-radius: 8px;
        font-size: 0.7rem;
        font-weight: 800;
        text-transform: uppercase;
        letter-spacing: 1px;
        z-index: 2;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        border: 2px solid white;
    }
    
    /* Make free badge smaller */
    .course-badge.free-badge {
        background: #10b981;
        color: white;
        padding: 4px 10px;
        font-size: 0.65rem;
    }

    .course-badge.free-badge i {
        font-size: 0.3rem;
        transform: scale(0.18);
        margin-right: 6px;
    }
    
    .course-badge.premium-badge {
        background: #f59e0b;
        color: white;
    }
    
    .course-content {
        padding: 24px;
    }
    
    .course-category {
        color: var(--ee-purple);
        font-size: 0.75rem;
        font-weight: 800;
        text-transform: uppercase;
        letter-spacing: 1px;
        margin-bottom: 12px;
        display: inline-block;
        padding: 4px 12px;
        background: rgba(124, 58, 237, 0.1);
        border-radius: 6px;
    }
    
    .course-card h3 {
        font-size: 1.25rem;
        font-weight: 800;
        margin-bottom: 12px;
        color: var(--ee-text);
        line-height: 1.4;
        display: -webkit-box;
        -webkit-line-clamp: 2;
        -webkit-box-orient: vertical;
        overflow: hidden;
    }
    
    .course-description {
        color: var(--ee-text-light);
        font-size: 0.9375rem;
        line-height: 1.6;
        margin-bottom: 16px;
        display: -webkit-box;
        -webkit-line-clamp: 2;
        -webkit-box-orient: vertical;
        overflow: hidden;
    }
    
    .course-meta {
        display: flex;
        gap: 20px;
        padding-top: 16px;
        border-top: 2px solid var(--ee-border);
        margin-top: 16px;
        font-size: 0.875rem;
        color: var(--ee-text-light);
    }
    
    .course-meta span {
        display: flex;
        align-items: center;
        gap: 8px;
    }
    
    .course-meta i {
        color: var(--ee-purple);
        font-size: 1rem;
    }
    
    .course-action {
        margin-top: 20px;
        width: 100%;
        padding: 14px;
        background: linear-gradient(135deg, var(--ee-purple), var(--ee-cyan));
        color: white;
        border: none;
        border-radius: 12px;
        font-weight: 700;
        font-size: 0.9375rem;
        cursor: pointer;
        transition: all 0.3s ease;
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 8px;
        text-decoration: none;
        text-align: center;
    }
    
    .course-action:hover {
        transform: translateY(-3px);
        box-shadow: 0 12px 32px var(--ee-glow);
    }
    
    .course-action.disabled {
        background: #94a3b8;
        cursor: not-allowed;
    }
    
    .course-action.disabled:hover {
        transform: none;
        box-shadow: none;
    }
    
    .no-courses {
        text-align: center;
        padding: 80px 20px;
        background: white;
        border-radius: 24px;
        border: 2px dashed var(--ee-border);
    }
    
    .no-courses i {
        font-size: 64px;
        color: var(--ee-border);
        margin-bottom: 24px;
    }
    
    .no-courses h3 {
        font-size: 1.5rem;
        color: var(--ee-text);
        margin-bottom: 12px;
    }
    
    .no-courses p {
        color: var(--ee-text-light);
        max-width: 500px;
        margin: 0 auto;
        line-height: 1.6;
    }
    
    .loading-spinner {
        text-align: center;
        padding: 80px 20px;
    }
    
    .spinner {
        border: 4px solid var(--ee-border);
        border-top: 4px solid var(--ee-purple);
        border-radius: 50%;
        width: 60px;
        height: 60px;
        animation: spin 1s linear infinite;
        margin: 0 auto 20px;
    }
    
    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }
    
    @media (max-width: 1024px) {
        .page-header h1 {
            font-size: 2.75rem;
        }
        
        .courses-filters {
            flex-direction: column;
            align-items: stretch;
            gap: 15px;
        }
        
        .search-box {
            min-width: 100%;
        }
        
        .language-selector,
        .category-filter-dropdown {
            width: 100%;
            justify-content: space-between;
        }
        
        .language-selector select,
        .category-filter-dropdown select {
            flex: 1;
        }
    }
    
    @media (max-width: 768px) {
        .page-header {
            padding: 80px 0 60px;
        }
        
        .page-header h1 {
            font-size: 2.25rem;
        }
        
        .page-header p {
            font-size: 1.125rem;
        }
        
        .courses-grid {
            grid-template-columns: 1fr;
        }
    }
    
    @media (max-width: 480px) {
        .page-header h1 {
            font-size: 1.875rem;
        }
        
        .filter-buttons {
            flex-direction: column;
        }
        
        .filter-btn {
            width: 100%;
        }
    }
</style>
@endpush

@section('content')
<!-- Page Header -->
<section class="page-header">
    <div class="container">
        <div class="page-header-content">
            <h1>Entrepreneur Training Courses</h1>
            <p>Simple, Easy, Enjoyable & Accessible digital learning for the next generation of entrepreneurs</p>
        </div>
    </div>
</section>

<!-- Courses Section -->
<section class="courses-section">
    <div class="container">
        <!-- Search & Filters -->
        <div class="courses-filters">
            <div class="search-box">
                <input type="text" id="searchInput" placeholder="🔍 Search courses, topics, or skills..." onkeyup="filterCourses()">
                <i class="fas fa-search"></i>
            </div>
            
            <!-- Language Selector -->
            <div id="languageSelector" class="language-selector" style="display: none;">
                <span class="language-selector-label">
                    <i class="fas fa-globe"></i> Language:
                </span>
                <select id="languageDropdown">
                    <option value="">All languages</option>
                </select>
            </div>
            
            <!-- Category Filter Dropdown -->
            <div class="category-filter-dropdown">
                <span class="category-filter-label">
                    <i class="fas fa-filter"></i> Category:
                </span>
                <select id="categoryDropdown" onchange="filterByCategory(this.value)">
                    <option value="all">All Courses</option>
                    <!-- Categories will be loaded dynamically -->
                </select>
            </div>
        </div>
        
        <!-- Course Count -->
        <div class="courses-count" id="coursesCount">⏳ Loading courses...</div>
        
        <!-- Loading Spinner -->
        <div id="loadingSpinner" class="loading-spinner">
            <div class="spinner"></div>
            <p style="color: var(--ee-text-light); font-weight: 600;">Loading our premium entrepreneur courses...</p>
        </div>
        
        <!-- Courses Grid -->
        <div id="coursesGrid" class="courses-grid" style="display: none;"></div>
        
        <!-- No Courses Message -->
        <div id="noCourses" class="no-courses" style="display: none;">
            <i class="fas fa-book-open"></i>
            <h3>No Courses Found 🧐</h3>
            <p>We couldn't find any courses matching your search. Try different keywords or browse our course categories.</p>
            <div style="margin-top: 24px;">
                <button class="filter-btn active" onclick="filterByCategory('all')" style="background: var(--ee-purple); color: white;">Show All Courses</button>
            </div>
        </div>
    </div>
</section>
@endsection

@push('scripts')
<script>
    let allCourses = [];
    let currentFilter = 'all';
    let selectedLanguageId = null;
    let multiLanguageEnabled = false;
    let availableLanguages = [];
    let availableCategories = [];
    
    // Load categories from API
    async function loadCategories() {
        try {
            const response = await fetch('/api/categories/public', {
                method: 'GET',
                headers: { 'Accept': 'application/json' }
            });
            
            if (response.ok) {
                const categories = await response.json();
                availableCategories = categories.filter(cat => cat.is_active)
                    .sort((a, b) => a.order - b.order);
                displayCategoryDropdown();
            }
        } catch (error) {
            console.error('Error loading categories:', error);
            // Keep default hardcoded categories if API fails
        }
    }
    
    // Display category dropdown with loaded categories
    function displayCategoryDropdown() {
        const dropdown = document.getElementById('categoryDropdown');
        if (!dropdown) return;
        
        // Clear existing options except "All Courses"
        dropdown.innerHTML = '<option value="all">All Courses</option>';
        
        // Add categories from database
        availableCategories.forEach(category => {
            const option = document.createElement('option');
            option.value = category.name.toLowerCase().replace(/\s+/g, '-');
            option.textContent = category.name;
            dropdown.appendChild(option);
        });
    }
    
    // Check if multi-language is enabled
    async function checkMultiLanguageSettings() {
        try {
            const response = await fetch('/api/settings', {
                method: 'GET',
                headers: { 'Accept': 'application/json' }
            });
            
            if (response.ok) {
                const data = await response.json();
                multiLanguageEnabled = data.settings?.enable_multi_language === 'true' || data.settings?.enable_multi_language === true;
            }
        } catch (error) {
            console.error('Error checking multi-language settings:', error);
        }
    }
    
    // Load available languages
    async function loadLanguages() {
        try {
            const response = await fetch('/api/languages/active', {
                method: 'GET',
                headers: { 'Accept': 'application/json' }
            });
            
            if (response.ok) {
                const data = await response.json();
                availableLanguages = data.languages || [];
                
                if (availableLanguages.length > 0) {
                    multiLanguageEnabled = true;
                    // Set default language or restore from localStorage
                    const savedLanguage = localStorage.getItem('preferred_language');
                    if (savedLanguage === 'all') {
                        selectedLanguageId = null;
                    } else if (savedLanguage) {
                        const parsed = parseInt(savedLanguage);
                        selectedLanguageId = isNaN(parsed) ? null : parsed;
                    } else {
                        selectedLanguageId = null;
                    }
                    displayLanguageSelector();
                }
            }
        } catch (error) {
            console.error('Error loading languages:', error);
        }
    }
    
    // Display language selector
    function displayLanguageSelector() {
        const selector = document.getElementById('languageSelector');
        const dropdown = document.getElementById('languageDropdown');
        
        if (!selector || !dropdown) {
            return;
        }
        
        // Always show if we have languages, even if just one
        if (!availableLanguages || availableLanguages.length === 0) {
            selector.style.display = 'none';
            return;
        }
        
        const options = [
            '<option value=\"\">All languages</option>',
            ...availableLanguages.map(lang => `<option value=\"${lang.id}\">${lang.flag_emoji || '🌐'} ${lang.name}</option>`)
        ].join('');
        dropdown.innerHTML = options;
        dropdown.value = selectedLanguageId || '';
        selector.style.display = 'flex';
        
        dropdown.onchange = (e) => {
            const val = e.target.value;
            selectLanguage(val === '' ? null : parseInt(val));
        };
    }
    
    // Select language
    function selectLanguage(languageId) {
        selectedLanguageId = languageId;
        localStorage.setItem('preferred_language', languageId === null ? 'all' : languageId);
        
        // Update dropdown selection
        const dropdown = document.getElementById('languageDropdown');
        if (dropdown) {
            dropdown.value = languageId || '';
        }
        
        // Reload courses with selected language
        loadCourses();
    }
    
    // Load courses from API
    async function loadCourses() {
        try {
            let url = '/api/courses/public';
            
            // Add language filter if multi-language is enabled
            if (selectedLanguageId) {
                url += `?language_id=${selectedLanguageId}`;
            }
            
            const response = await fetch(url, {
                method: 'GET',
                headers: {
                    'Accept': 'application/json',
                }
            });
            
            if (!response.ok) {
                throw new Error('Failed to load courses');
            }
            
            const data = await response.json();
            allCourses = data.courses || [];
            
            document.getElementById('loadingSpinner').style.display = 'none';
            displayCourses(allCourses);
            
        } catch (error) {
            console.error('Error loading courses:', error);
            document.getElementById('loadingSpinner').style.display = 'none';
            
            // Show no courses message with error details
            const noCoursesDiv = document.getElementById('noCourses');
            noCoursesDiv.style.display = 'block';
            noCoursesDiv.innerHTML = `
                <i class="fas fa-exclamation-triangle" style="color: #f59e0b;"></i>
                <h3>Unable to Load Courses</h3>
                <p>There was an error loading the courses. This might be because:</p>
                <ul style="text-align: left; max-width: 500px; margin: 20px auto; line-height: 2; color: var(--ee-text-light);">
                    <li>No courses have been created yet by the admin</li>
                    <li>The database connection needs to be configured</li>
                    <li>You need to log in as admin and create courses</li>
                </ul>
                <p style="margin-top: 20px; color: var(--ee-text-light);">Please contact the administrator or try again later.</p>
            `;
            document.getElementById('coursesCount').textContent = '0 courses available';
        }
    }
    
    function calculateDiscountedPrice(price, discount, discountType) {
        const priceValue = parseFloat(price || 0);
        const discountValue = parseFloat(discount || 0);
        if (priceValue <= 0) return 0;
        if (discountValue <= 0) return priceValue;
        const type = discountType === 'percent' ? 'percent' : 'amount';
        const discounted = type === 'percent'
            ? priceValue - (priceValue * (discountValue / 100))
            : priceValue - discountValue;
        return Math.max(discounted, 0);
    }
    
    // Display courses
    function displayCourses(courses) {
        const grid = document.getElementById('coursesGrid');
        const noCoursesMsg = document.getElementById('noCourses');
        const countElement = document.getElementById('coursesCount');
        
        if (courses.length === 0) {
            grid.style.display = 'none';
            noCoursesMsg.style.display = 'block';
            countElement.textContent = 'No courses found 🫤';
            countElement.style.color = '#f59e0b';
            return;
        }
        
        grid.style.display = 'grid';
        noCoursesMsg.style.display = 'none';
        
        const courseWord = courses.length === 1 ? 'course' : 'courses';
        countElement.textContent = `🎯 Found ${courses.length} ${courseWord}`;
        countElement.style.color = 'var(--ee-purple)';
        
        grid.innerHTML = courses.map(course => {
            const isAuthenticated = {{ auth()->check() ? 'true' : 'false' }};
            const courseImage = course.thumbnail_url || course.image_url;
            const currencyCode = (course.currency || 'USD').toUpperCase();
            const priceValue = parseFloat(course.price ?? 0);
            const discountValue = parseFloat(course.discount ?? 0);
            const discountType = course.discount_type || 'amount';
            const discountedPrice = parseFloat(course.discounted_price ?? calculateDiscountedPrice(priceValue, discountValue, discountType));
            const hasDiscount = !course.is_free && discountValue > 0 && priceValue > 0 && discountedPrice < priceValue;
            const discountLabel = hasDiscount
                ? (discountType === 'percent' ? `${discountValue}% off` : `${currencyCode} ${discountValue.toFixed(2)} off`)
                : '';
            
            // Get category from course data (from database)
            let category = course.category || 'General';
            
            const originalPriceText = `${currencyCode} ${priceValue.toFixed(2)}`;
            const finalPriceText = `${currencyCode} ${(hasDiscount ? discountedPrice : priceValue).toFixed(2)}`;
            
            // Determine enrollment status and button
            let actionButton = '';
            if (isAuthenticated) {
                if (course.is_free) {
                    actionButton = `
                        <a href="/learn/course/${course.id}" class="course-action">
                            <i class="fas fa-play-circle"></i> Start Learning Free
                        </a>
                    `;
                } else {
                    actionButton = `
                        <a href="/payment/checkout/${course.id}" class="course-action">
                            <i class="fas fa-shopping-cart"></i> Purchase - ${finalPriceText}${hasDiscount ? ` <span style="text-decoration: line-through; color: rgba(255,255,255,0.7); margin-left: 8px; font-size: 0.875rem;">${originalPriceText}</span>` : ''}
                        </a>
                    `;
                }
            } else {
                actionButton = `
                    <a href="/login?redirect=/learn/course/${course.id}" class="course-action">
                        <i class="fas fa-sign-in-alt"></i> Login to Access Course
                    </a>
                `;
            }

            return `
                <div class="course-card" data-category="${category}">
                    <div class="course-image">
                        ${courseImage
                            ? `<img src="${courseImage}" alt="${escapeHtml(course.name)} cover image">`
                            : '<div class="course-image-placeholder"><i class="fas fa-graduation-cap"></i></div>'
                        }
                        ${course.is_free 
                            ? '<div class="course-badge free-badge"><i class="fas fa-gift"></i> FREE</div>' 
                            : `<div class="course-badge premium-badge" ${hasDiscount ? 'style="background:#0f172a;color:#f8fafc;"' : ''}>
                                  <i class="fas fa-credit-card"></i>
                                  ${hasDiscount
                                    ? `<span style="text-decoration: line-through; opacity: 0.85;">${originalPriceText}</span> <strong style="margin-left: 6px;">${currencyCode} ${discountedPrice.toFixed(2)}</strong>`
                                    : originalPriceText}
                               </div>
                               ${hasDiscount ? `<div class="course-badge" style="left: 15px; right: auto; background:#f97316; color:white; font-weight:800;"><i class="fas fa-bolt"></i> ${discountLabel}</div>` : ''}`
                        }
                    </div>
                    <div class="course-content">
                        <div style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 12px;">
                            <div class="course-category">${category}</div>
                            ${course.language ? `
                                <div style="font-size: 0.875rem; color: var(--ee-text-light); display: flex; align-items: center; gap: 4px;">
                                    <span>${course.language.flag_emoji}</span>
                                    <span style="font-weight: 600;">${course.language.name}</span>
                                </div>
                            ` : ''}
                        </div>
                        <h3>${escapeHtml(course.name)}</h3>
                        <p class="course-description">${escapeHtml(course.short_description || 'Comprehensive training course covering essential entrepreneur skills and strategies for business growth.')}</p>
                        <div class="course-meta">
                            <span><i class="fas fa-book"></i> ${course.lessons_count || 0} Lessons</span>
                            <span><i class="fas fa-clock"></i> Self-paced</span>
                            ${course.is_free ? '' : `<span><i class="fas fa-tag"></i> ${hasDiscount
                                ? `<span style="text-decoration: line-through; color: #9ca3af;">${originalPriceText}</span> <span style="color:#10b981; font-weight: 700; margin-left: 6px;">${currencyCode} ${discountedPrice.toFixed(2)}</span>`
                                : finalPriceText}</span>`}
                        </div>
                        ${actionButton}
                    </div>
                </div>
            `;
        }).join('');
    }
    
    // Filter by category
    function filterByCategory(category) {
        currentFilter = category;
        
        // Update dropdown value
        const dropdown = document.getElementById('categoryDropdown');
        if (dropdown) {
            dropdown.value = category;
        }
        
        // Filter courses
        let filtered = allCourses;
        
        if (category !== 'all') {
            filtered = filtered.filter(course => {
                const courseCategory = (course.category || '').toLowerCase().replace(/\s+/g, '-');
                return courseCategory === category;
            });
        }
        
        // Apply search filter if active
        const searchTerm = document.getElementById('searchInput').value.toLowerCase();
        if (searchTerm) {
            filtered = filtered.filter(course => 
                course.name.toLowerCase().includes(searchTerm) ||
                (course.short_description && course.short_description.toLowerCase().includes(searchTerm))
            );
        }
        
        displayCourses(filtered);
    }
    
    // Initialize page
    async function initializePage() {
        console.log('Initializing courses page...');
        try {
            await checkMultiLanguageSettings();
            await loadLanguages();
            await loadCategories();
            await loadCourses();
            console.log('Courses page initialized successfully');
        } catch (error) {
            console.error('Error initializing page:', error);
            document.getElementById('loadingSpinner').style.display = 'none';
            document.getElementById('noCourses').style.display = 'block';
        }
    }
    
    // Start when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initializePage);
    } else {
        initializePage();
    }
    
    // Search filter
    function filterCourses() {
        const searchTerm = document.getElementById('searchInput').value.toLowerCase();
        
        let filtered = allCourses;
        
        // Apply category filter
        if (currentFilter !== 'all') {
            filtered = filtered.filter(course => {
                const courseCategory = (course.category || '').toLowerCase().replace(/\s+/g, '-');
                return courseCategory === currentFilter;
            });
        }
        
        // Apply search filter
        if (searchTerm) {
            filtered = filtered.filter(course => 
                course.name.toLowerCase().includes(searchTerm) ||
                (course.short_description && course.short_description.toLowerCase().includes(searchTerm))
            );
        }
        
        displayCourses(filtered);
    }
    
    // Escape HTML to prevent XSS
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
</script>
@endpush
