<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, user-scalable=no">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>{{ $topic->title }} - {{ $course->name }}</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            -webkit-tap-highlight-color: transparent;
        }

        :root {
            --primary: #667eea;
            --primary-dark: #5568d3;
            --secondary: #764ba2;
            --success: #10b981;
            --danger: #ef4444;
            --text-dark: #1f2937;
            --text-gray: #6b7280;
            --bg-light: #f9fafb;
            --card-shadow: 0 10px 30px rgba(0,0,0,0.08);
            --card-shadow-hover: 0 20px 40px rgba(0,0,0,0.12);
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            overflow-x: hidden;
            -webkit-font-smoothing: antialiased;
        }

        /* Navbar */
        .navbar {
            background: rgba(255,255,255,0.98);
            backdrop-filter: blur(20px);
            box-shadow: 0 2px 20px rgba(0,0,0,0.08);
            position: sticky;
            top: 0;
            z-index: 100;
        }

        .navbar-content {
            max-width: 900px;
            margin: 0 auto;
            padding: 16px 20px;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .btn-back {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 10px 18px;
            background: rgba(102, 126, 234, 0.1);
            color: var(--primary);
            text-decoration: none;
            border-radius: 12px;
            font-weight: 600;
            font-size: 0.9em;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }

        .btn-back:hover {
            background: var(--primary);
            color: white;
            transform: translateX(-4px);
        }

        .btn-back:active {
            transform: translateX(-4px) scale(0.95);
        }

        /* Progress Badge */
        .progress-badge {
            background: linear-gradient(135deg, var(--success), #059669);
            color: white;
            padding: 8px 16px;
            border-radius: 12px;
            font-weight: 700;
            font-size: 0.85em;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        /* Container */
        .container {
            max-width: 900px;
            margin: 0 auto;
            padding: 24px 20px 100px;
        }

        /* Breadcrumb */
        .breadcrumb {
            background: rgba(255,255,255,0.15);
            backdrop-filter: blur(10px);
            padding: 12px 20px;
            border-radius: 12px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
            flex-wrap: wrap;
            font-size: 0.85em;
        }

        .breadcrumb a {
            color: white;
            text-decoration: none;
            font-weight: 600;
            transition: opacity 0.2s;
        }

        .breadcrumb a:hover {
            opacity: 0.8;
        }

        .breadcrumb span {
            color: rgba(255,255,255,0.7);
        }

        /* Topic Header */
        .topic-header {
            background: white;
            padding: 28px;
            border-radius: 20px;
            margin-bottom: 20px;
            box-shadow: var(--card-shadow);
        }

        .topic-title {
            font-size: 1.8em;
            font-weight: 800;
            color: var(--text-dark);
            margin-bottom: 16px;
            line-height: 1.3;
        }

        .topic-meta {
            display: flex;
            gap: 12px;
            flex-wrap: wrap;
        }

        .meta-badge {
            padding: 6px 14px;
            border-radius: 10px;
            font-size: 0.8em;
            font-weight: 600;
            display: inline-flex;
            align-items: center;
            gap: 6px;
        }

        .meta-badge.completed {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
        }

        .meta-badge.in-progress {
            background: rgba(102, 126, 234, 0.1);
            color: var(--primary);
        }

        /* Progress Bar */
        .progress-bar-container {
            margin-top: 16px;
        }

        .progress-bar {
            height: 6px;
            background: rgba(102, 126, 234, 0.1);
            border-radius: 10px;
            overflow: hidden;
            margin-bottom: 8px;
        }

        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
            border-radius: 10px;
            transition: width 0.6s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
        }

        .progress-fill::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            bottom: 0;
            right: 0;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
            animation: shimmer 2s infinite;
        }

        @keyframes shimmer {
            0% { transform: translateX(-100%); }
            100% { transform: translateX(100%); }
        }

        .progress-text {
            font-size: 0.85em;
            color: var(--text-gray);
            font-weight: 600;
        }

        /* Content Card - The Main Star */
        .content-card {
            background: white;
            border-radius: 24px;
            box-shadow: var(--card-shadow);
            overflow: hidden;
            margin-bottom: 20px;
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            animation: slideUp 0.5s ease-out;
        }

        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .content-card:hover {
            box-shadow: var(--card-shadow-hover);
            transform: translateY(-2px);
        }

        /* Stage Indicator */
        .stage-indicator {
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            padding: 16px 24px;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .stage-name {
            font-weight: 700;
            font-size: 0.9em;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .stage-icon {
            font-size: 1.4em;
        }

        .stage-counter {
            background: rgba(255,255,255,0.2);
            padding: 4px 12px;
            border-radius: 8px;
            font-weight: 700;
            font-size: 0.85em;
        }

        /* Content Body */
        .content-body {
            padding: 32px 28px;
            min-height: 300px;
        }

        /* Text Content */
        .text-content {
            color: var(--text-dark);
            font-size: 1.05em;
            line-height: 1.8;
            font-weight: 400;
        }

        .text-content h1, .text-content h2, .text-content h3 {
            margin: 24px 0 16px;
            font-weight: 700;
            color: var(--text-dark);
        }

        .text-content p {
            margin-bottom: 16px;
        }

        .text-content ul, .text-content ol {
            margin-left: 24px;
            margin-bottom: 16px;
        }

        .text-content li {
            margin-bottom: 8px;
        }

        .text-content strong {
            font-weight: 700;
            color: var(--text-dark);
        }

        .text-content a {
            color: var(--primary);
            text-decoration: underline;
        }

        /* Image Display */
        .image-display {
            text-align: center;
        }

        .image-display img {
            max-width: 100%;
            height: auto;
            border-radius: 16px;
            box-shadow: 0 8px 24px rgba(0,0,0,0.1);
            transition: transform 0.3s ease;
            cursor: pointer;
        }

        .image-display img:hover {
            transform: scale(1.02);
        }

        .image-caption {
            margin-top: 16px;
            font-size: 0.9em;
            color: var(--text-gray);
            font-style: italic;
        }

        /* Video Display */
        .video-display {
            text-align: center;
        }

        .video-wrapper {
            position: relative;
            padding-bottom: 56.25%; /* 16:9 ratio */
            height: 0;
            border-radius: 16px;
            overflow: hidden;
            box-shadow: 0 8px 24px rgba(0,0,0,0.15);
        }

        .video-wrapper iframe,
        .video-wrapper video {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            border-radius: 16px;
        }

        .video-controls {
            margin-top: 20px;
            display: flex;
            justify-content: center;
        }

        /* Navigation Buttons */
        .navigation-buttons {
            display: flex;
            gap: 12px;
            padding: 24px 28px;
            border-top: 1px solid #f3f4f6;
            background: #fafafa;
        }

        .nav-btn {
            flex: 1;
            padding: 16px 24px;
            border: none;
            border-radius: 14px;
            font-weight: 700;
            font-size: 0.95em;
            cursor: pointer;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
            text-decoration: none;
            position: relative;
            overflow: hidden;
        }

        .nav-btn::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 0;
            height: 0;
            border-radius: 50%;
            background: rgba(255,255,255,0.3);
            transform: translate(-50%, -50%);
            transition: width 0.6s, height 0.6s;
        }

        .nav-btn:active::before {
            width: 300px;
            height: 300px;
        }

        .nav-btn.btn-prev {
            background: #e5e7eb;
            color: var(--text-dark);
        }

        .nav-btn.btn-prev:hover {
            background: #d1d5db;
            transform: translateX(-4px);
        }

        .nav-btn.btn-prev:disabled {
            opacity: 0.4;
            cursor: not-allowed;
            transform: none;
        }

        .nav-btn.btn-next {
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
        }

        .nav-btn.btn-next:hover {
            box-shadow: 0 8px 24px rgba(102, 126, 234, 0.4);
            transform: translateY(-2px);
        }

        .nav-btn.btn-next:active {
            transform: translateY(0);
        }

        /* Loading State */
        .loading {
            text-align: center;
            padding: 60px 20px;
        }

        .loading i {
            font-size: 2.5em;
            color: var(--primary);
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }

        .loading p {
            margin-top: 16px;
            color: var(--text-gray);
            font-weight: 600;
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--text-gray);
        }

        .empty-state i {
            font-size: 3em;
            margin-bottom: 16px;
            opacity: 0.5;
        }

        /* Toast Notification */
        .toast {
            position: fixed;
            top: 24px;
            right: 24px;
            background: linear-gradient(135deg, var(--success), #059669);
            color: white;
            padding: 16px 24px;
            border-radius: 14px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
            z-index: 9999;
            font-weight: 600;
            animation: slideInRight 0.4s ease;
            max-width: 90%;
        }

        .toast.error {
            background: linear-gradient(135deg, var(--danger), #dc2626);
        }

        @keyframes slideInRight {
            from {
                transform: translateX(400px);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }

        @keyframes slideOutRight {
            from {
                transform: translateX(0);
                opacity: 1;
            }
            to {
                transform: translateX(400px);
                opacity: 0;
            }
        }

        /* Mobile Responsive */
        @media (max-width: 768px) {
            .navbar-content {
                padding: 14px 16px;
            }

            .container {
                padding: 20px 16px 100px;
            }

            .topic-header {
                padding: 20px;
            }

            .topic-title {
                font-size: 1.5em;
            }

            .content-body {
                padding: 24px 20px;
                min-height: 250px;
            }

            .text-content {
                font-size: 1em;
            }

            .navigation-buttons {
                padding: 20px 16px;
                flex-direction: column;
            }

            .nav-btn {
                padding: 14px 20px;
            }

            .stage-indicator {
                padding: 14px 20px;
            }

            .btn-back {
                font-size: 0.85em;
                padding: 8px 14px;
            }

            .progress-badge {
                font-size: 0.8em;
                padding: 6px 12px;
            }
        }

        @media (max-width: 480px) {
            .topic-title {
                font-size: 1.3em;
            }

            .breadcrumb {
                font-size: 0.8em;
                padding: 10px 16px;
            }

            .content-body {
                padding: 20px 16px;
            }

            .text-content {
                font-size: 0.95em;
                line-height: 1.7;
            }

            .toast {
                right: 16px;
                top: 16px;
                left: 16px;
                max-width: none;
            }
        }

        /* Focus Mode - Hide distractions on scroll */
        .focus-mode .navbar {
            transform: translateY(-100%);
            transition: transform 0.3s ease;
        }

        /* Smooth transitions everywhere */
        * {
            transition: background-color 0.2s ease, color 0.2s ease, border-color 0.2s ease;
        }
    </style>
</head>
<body>
    <!-- Navbar -->
    <nav class="navbar">
        <div class="navbar-content">
            <a href="{{ route('learn.course', $course->id) }}" class="btn-back">
                <i class="fas fa-arrow-left"></i>
                <span>Back to Course</span>
            </a>
            <div class="progress-badge">
                <i class="fas fa-chart-line"></i>
                <span id="overall-progress">0%</span>
            </div>
        </div>
    </nav>

    <div class="container">
        <!-- Breadcrumb -->
        <div class="breadcrumb">
            <a href="{{ route('dashboard') }}">Dashboard</a>
            <span>/</span>
            <a href="{{ route('learn.course', $course->id) }}">{{ $course->name }}</a>
            <span>/</span>
            <span>{{ $lesson->title }}</span>
        </div>

        <!-- Topic Header -->
        <div class="topic-header">
            <h1 class="topic-title">{{ $topic->title }}</h1>
            <div class="topic-meta">
                @if($userProgress && $userProgress->completed)
                    <span class="meta-badge completed">
                        <i class="fas fa-check-circle"></i>
                        Completed
                    </span>
                @else
                    <span class="meta-badge in-progress">
                        <i class="fas fa-spinner"></i>
                        In Progress
                    </span>
                @endif
                @if($quizQuestions->count() > 0)
                    <span class="meta-badge" style="background: rgba(102, 126, 234, 0.1); color: var(--primary);">
                        <i class="fas fa-question-circle"></i>
                        {{ $quizQuestions->count() }} Quiz Questions
                    </span>
                @endif
            </div>
            
            <div class="progress-bar-container">
                <div class="progress-bar">
                    <div class="progress-fill" id="stage-progress" style="width: 0%"></div>
                </div>
                <div class="progress-text" id="progress-text">Loading...</div>
            </div>
        </div>

        <!-- Main Content Card -->
        <div class="content-card">
            <!-- Stage Indicator -->
            <div class="stage-indicator">
                <div class="stage-name">
                    <span class="stage-icon" id="stage-icon">📝</span>
                    <span id="stage-name">Text Content</span>
                </div>
                <div class="stage-counter" id="stage-counter">1 of 3</div>
            </div>

            <!-- Content Body -->
            <div class="content-body" id="content-body">
                <div class="loading">
                    <i class="fas fa-spinner"></i>
                    <p>Loading content...</p>
                </div>
            </div>

            <!-- Navigation Buttons -->
            <div class="navigation-buttons">
                <button class="nav-btn btn-prev" id="btn-prev" disabled>
                    <i class="fas fa-arrow-left"></i>
                    <span>Previous</span>
                </button>
                <button class="nav-btn btn-next" id="btn-next">
                    <span>Next</span>
                    <i class="fas fa-arrow-right"></i>
                </button>
            </div>
        </div>
    </div>

    <script>
        const topicId = {{ $topic->id }};
        const courseId = {{ $course->id }};
        const lessonId = {{ $lesson->id }};
        const csrfToken = document.querySelector('meta[name="csrf-token"]').content;

        // Topic data
        const topicData = {
            id: {{ $topic->id }},
            title: "{{ $topic->title }}",
            body: `{!! addslashes($topic->body ?? '') !!}`,
            image_url: "{{ $topic->image_url ?? '' }}",
            video_url: "{{ $topic->video_url ?? '' }}",
            hasQuiz: {{ $quizQuestions->count() > 0 ? 'true' : 'false' }}
        };

        const nextTopicUrl = "{{ $nextTopic ? route('learn.topic', ['course' => $course->id, 'lesson' => $nextTopic->lesson_id, 'topic' => $nextTopic->id]) : '' }}";
        const prevTopicUrl = "{{ $previousTopic ? route('learn.topic', ['course' => $course->id, 'lesson' => $previousTopic->lesson_id, 'topic' => $previousTopic->id]) : '' }}";
        const courseUrl = "{{ route('learn.course', $course->id) }}";
        
        const isLastTopicInLesson = {{ $isLastTopicInLesson ? 'true' : 'false' }};
        const hasNextLesson = {{ isset($nextLesson) && $nextLesson ? 'true' : 'false' }};
        
        @if(isset($nextLesson) && $nextLesson)
        const nextLessonData = {
            id: {{ $nextLesson->id }},
            title: "{{ $nextLesson->title }}",
            description: "{{ $nextLesson->intro ?? '' }}"
        };
        @endif

        // App State
        const state = {
            stages: [],
            currentStageIndex: 0,
            textChunks: [],
            currentChunkIndex: 0,
            completed: {{ $userProgress && $userProgress->completed ? 'true' : 'false' }}
        };

        // Initialize
        document.addEventListener('DOMContentLoaded', () => {
            initializeStages();
            renderCurrentStage();
            updateProgress();
        });

        function initializeStages() {
            // Build stages based on available content
            if (topicData.body && topicData.body.trim()) {
                // Split text into chunks (max 500 words per chunk)
                const chunks = chunkText(topicData.body, 500);
                chunks.forEach((chunk, index) => {
                    state.stages.push({
                        type: 'text',
                        content: chunk,
                        chunkIndex: index,
                        totalChunks: chunks.length
                    });
                });
                state.textChunks = chunks;
            }

            if (topicData.image_url && topicData.image_url.trim()) {
                state.stages.push({
                    type: 'image',
                    content: topicData.image_url
                });
            }

            if (topicData.video_url && topicData.video_url.trim()) {
                state.stages.push({
                    type: 'video',
                    content: topicData.video_url
                });
            }

            // If no content, show empty state
            if (state.stages.length === 0) {
                state.stages.push({
                    type: 'empty',
                    content: 'No content available for this topic.'
                });
            }
        }

        function chunkText(text, maxWords) {
            // Simple HTML-aware chunking
            const hasHTML = /<[a-z][\s\S]*>/i.test(text);
            
            if (hasHTML) {
                // Don't chunk HTML content
                return [text];
            }

            // Split by words
            const words = text.split(/\s+/);
            const chunks = [];
            
            for (let i = 0; i < words.length; i += maxWords) {
                chunks.push(words.slice(i, i + maxWords).join(' '));
            }
            
            return chunks.length > 0 ? chunks : [text];
        }

        function renderCurrentStage() {
            const stage = state.stages[state.currentStageIndex];
            const contentBody = document.getElementById('content-body');
            const stageIcon = document.getElementById('stage-icon');
            const stageName = document.getElementById('stage-name');
            const stageCounter = document.getElementById('stage-counter');
            
            // Update stage indicator
            stageCounter.textContent = `${state.currentStageIndex + 1} of ${state.stages.length}`;
            
            // Render based on type
            switch (stage.type) {
                case 'text':
                    stageIcon.textContent = '📝';
                    stageName.textContent = stage.totalChunks > 1 
                        ? `Text Content (Part ${stage.chunkIndex + 1}/${stage.totalChunks})` 
                        : 'Text Content';
                    contentBody.innerHTML = `<div class="text-content">${stage.content}</div>`;
                    break;
                    
                case 'image':
                    stageIcon.textContent = '🖼️';
                    stageName.textContent = 'Image';
                    contentBody.innerHTML = `
                        <div class="image-display">
                            <img src="${stage.content}" alt="Topic Image" onload="this.style.opacity=1" style="opacity:0; transition: opacity 0.3s;">
                        </div>
                    `;
                    break;
                    
                case 'video':
                    stageIcon.textContent = '🎥';
                    stageName.textContent = 'Video';
                    contentBody.innerHTML = `
                        <div class="video-display">
                            <div class="video-wrapper">
                                ${getVideoEmbed(stage.content)}
                            </div>
                        </div>
                    `;
                    break;
                    
                case 'empty':
                    stageIcon.textContent = '📭';
                    stageName.textContent = 'No Content';
                    contentBody.innerHTML = `
                        <div class="empty-state">
                            <i class="fas fa-inbox"></i>
                            <p>${stage.content}</p>
                        </div>
                    `;
                    break;
            }
            
            // Update button states
            updateButtonStates();
            
            // Trigger animation
            contentBody.style.animation = 'none';
            setTimeout(() => {
                contentBody.style.animation = 'slideUp 0.5s ease-out';
            }, 10);
        }

        function getVideoEmbed(url) {
            // YouTube
            if (/youtu\.be|youtube\.com/.test(url)) {
                const id = (url.match(/(?:v=|\/)([A-Za-z0-9_-]{11})/)||[])[1] || '';
                return `<iframe src="https://www.youtube.com/embed/${id}?rel=0&modestbranding=1" 
                        frameborder="0" allowfullscreen></iframe>`;
            }
            // Regular video
            return `<video controls style="width:100%;border-radius:16px;">
                    <source src="${url}" type="video/mp4">
                    Your browser does not support the video tag.
                    </video>`;
        }

        function updateButtonStates() {
            const btnPrev = document.getElementById('btn-prev');
            const btnNext = document.getElementById('btn-next');
            
            // Previous button
            if (state.currentStageIndex === 0) {
                btnPrev.disabled = true;
                if (prevTopicUrl) {
                    btnPrev.innerHTML = '<i class="fas fa-arrow-left"></i><span>Previous Topic</span>';
                    btnPrev.onclick = () => window.location.href = prevTopicUrl;
                    btnPrev.disabled = false;
                }
            } else {
                btnPrev.disabled = false;
                btnPrev.innerHTML = '<i class="fas fa-arrow-left"></i><span>Previous</span>';
                btnPrev.onclick = handlePrevious;
            }
            
            // Next button
            if (state.currentStageIndex === state.stages.length - 1) {
                btnNext.innerHTML = '<span>Complete & Next</span><i class="fas fa-check"></i>';
                btnNext.onclick = handleComplete;
            } else {
                btnNext.innerHTML = '<span>Next</span><i class="fas fa-arrow-right"></i>';
                btnNext.onclick = handleNext;
            }
        }

        function handlePrevious() {
            if (state.currentStageIndex > 0) {
                state.currentStageIndex--;
                renderCurrentStage();
                updateProgress();
            }
        }

        function handleNext() {
            if (state.currentStageIndex < state.stages.length - 1) {
                state.currentStageIndex++;
                renderCurrentStage();
                updateProgress();
            }
        }

        async function handleComplete() {
            const btnNext = document.getElementById('btn-next');
            btnNext.disabled = true;
            btnNext.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span>Completing...</span>';
            
            try {
                const response = await fetch(`/learn/topic/${topicId}/complete`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': csrfToken
                    }
                });
                
                const data = await response.json();
                
                if (data.success) {
                    state.completed = true;
                    showToast('Topic completed!', 'success');
                    
                    // Check if we need to show congratulations modal
                    if (isLastTopicInLesson && hasNextLesson) {
                        setTimeout(async () => {
                            const lessonCheck = await fetch(`/api/lesson/${lessonId}/check-completion`, {
                                headers: { 'X-CSRF-TOKEN': csrfToken }
                            });
                            const lessonData = await lessonCheck.json();
                            
                            if (lessonData.completed) {
                                // Show congratulations modal (implement this)
                                showCongratsModal();
                                return;
                            }
                            
                            // Navigate to next topic or course
                            if (nextTopicUrl) {
                                window.location.href = nextTopicUrl;
                            } else {
                                window.location.href = courseUrl;
                            }
                        }, 500);
                    } else {
                        // Navigate to next topic or course
                        setTimeout(() => {
                            if (nextTopicUrl) {
                                window.location.href = nextTopicUrl;
                            } else {
                                window.location.href = courseUrl;
                            }
                        }, 1000);
                    }
                } else {
                    throw new Error(data.message || 'Failed to complete topic');
                }
            } catch (error) {
                showToast(error.message, 'error');
                btnNext.disabled = false;
                updateButtonStates();
            }
        }

        function updateProgress() {
            const progress = ((state.currentStageIndex + 1) / state.stages.length) * 100;
            document.getElementById('stage-progress').style.width = progress + '%';
            document.getElementById('progress-text').textContent = 
                `Stage ${state.currentStageIndex + 1} of ${state.stages.length}`;
        }

        function showToast(message, type = 'success') {
            const toast = document.createElement('div');
            toast.className = `toast ${type}`;
            toast.textContent = message;
            document.body.appendChild(toast);
            
            setTimeout(() => {
                toast.style.animation = 'slideOutRight 0.4s ease';
                setTimeout(() => toast.remove(), 400);
            }, 4000);
        }

        function showCongratsModal() {
            // This will integrate with existing modal system
            showToast('Lesson completed! 🎉', 'success');
            setTimeout(() => {
                window.location.href = courseUrl;
            }, 2000);
        }

        // Focus mode - hide navbar on scroll down
        let lastScroll = 0;
        window.addEventListener('scroll', () => {
            const currentScroll = window.pageYOffset;
            if (currentScroll > lastScroll && currentScroll > 100) {
                document.body.classList.add('focus-mode');
            } else {
                document.body.classList.remove('focus-mode');
            }
            lastScroll = currentScroll;
        });

        // Update overall progress badge
        fetch(`/api/course/${courseId}/progress`)
            .then(r => r.json())
            .then(d => {
                if (d.progress !== undefined) {
                    document.getElementById('overall-progress').textContent = d.progress + '%';
                }
            })
            .catch(() => {});
    </script>
</body>
</html>
