<?php

use App\Http\Controllers\AuthController;
use App\Http\Controllers\CourseController;
use App\Http\Controllers\LessonController;
use App\Http\Controllers\TopicController;
use App\Http\Controllers\QuizController;
use App\Http\Controllers\ProgressController;
use App\Http\Controllers\SettingsController;
use App\Http\Controllers\StudyTimeController;
use App\Http\Controllers\SurveyController;
use App\Http\Controllers\CategoryController;
use App\Http\Controllers\Api\SettingsController as ApiSettingsController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

Route::middleware('auth:sanctum')->get('/user', function (Request $request) {
    return $request->user();
});

// Authentication routes
Route::post('/login', [AuthController::class, 'login']);
Route::post('/register', [AuthController::class, 'register']);
Route::post('/logout', [AuthController::class, 'logout']);
Route::get('/session', [AuthController::class, 'session']);

// Public routes (no authentication required) - MUST be before auth:sanctum group
Route::get('/courses/public', [CourseController::class, 'publicIndex']);
Route::get('/categories/public', [CategoryController::class, 'index']);

// Payment webhooks (no auth required)
Route::post('/webhooks/stripe', [App\Http\Controllers\WebhookController::class, 'handleStripe']);
Route::post('/webhooks/paypal', [App\Http\Controllers\WebhookController::class, 'handlePayPal']);
Route::post('/webhooks/paystack', [App\Http\Controllers\WebhookController::class, 'handlePaystack']);

// Protected routes (require authentication)
Route::middleware(['auth:sanctum'])->group(function () {

    // Category routes
    Route::get('/categories', [CategoryController::class, 'index']);
    Route::post('/categories', [CategoryController::class, 'store']);
    Route::get('/categories/{category}', [CategoryController::class, 'show']);
    Route::put('/categories/{category}', [CategoryController::class, 'update']);
    Route::delete('/categories/{category}', [CategoryController::class, 'destroy']);

    // Course routes
    Route::get('/courses', [CourseController::class, 'index']);
    Route::post('/courses', [CourseController::class, 'store']);
    Route::get('/courses/{id}', [CourseController::class, 'show']);
    Route::put('/courses/{id}', [CourseController::class, 'update']);
    Route::delete('/courses/{id}', [CourseController::class, 'destroy']);

    // Lesson routes (formerly sections)
    Route::get('/lessons', [LessonController::class, 'index']); // ?course_id=123
    Route::post('/lessons', [LessonController::class, 'store']);
    Route::get('/lessons/{id}', [LessonController::class, 'show']);
    Route::put('/lessons/{id}', [LessonController::class, 'update']);
    Route::delete('/lessons/{id}', [LessonController::class, 'destroy']);

    // Topic routes
    Route::get('/topics', [TopicController::class, 'index']); // ?lesson_id=123
    Route::post('/topics', [TopicController::class, 'store']);
    Route::get('/topics/{id}', [TopicController::class, 'show']);
    Route::put('/topics/{id}', [TopicController::class, 'update']);
    Route::delete('/topics/{id}', [TopicController::class, 'destroy']);

    // Quiz routes
    Route::get('/quiz', [QuizController::class, 'getQuiz']); // ?topic_id=123
    Route::get('/lesson-quiz', [QuizController::class, 'getLessonQuiz']); // ?lesson_id=123
    Route::get('/quiz-questions', [QuizController::class, 'index']); // ?lesson_id=123 or ?topic_id=123
    Route::post('/quiz-questions', [QuizController::class, 'store']);
    Route::put('/quiz-questions/{id}', [QuizController::class, 'update']);
    Route::delete('/quiz-questions/{id}', [QuizController::class, 'destroy']);

    // Progress routes
    Route::post('/update-progress', [ProgressController::class, 'updateProgress']);
    Route::post('/update-lesson-progress', [ProgressController::class, 'updateLessonProgress']);
    Route::get('/progress', [ProgressController::class, 'getProgress']); // ?topic_id=123
    Route::get('/lesson-progress', [ProgressController::class, 'getLessonProgress']); // ?lesson_id=123
    Route::get('/user-progress', [ProgressController::class, 'getUserProgress']);
    Route::get('/resume-point', [ProgressController::class, 'getResumePoint']); // ?course_id=123
    
    // Course progress route (for frontend)
    Route::get('/course/{courseId}/progress', function ($courseId) {
        /** @var \App\Models\User $user */
        $user = auth()->user();
        
        $enrollment = \App\Models\CourseEnrollment::where('user_id', $user->id)
            ->where('course_id', $courseId)
            ->whereIn('status', ['enrolled', 'in-progress', 'completed'])
            ->first();
        
        if (!$enrollment) {
            return response()->json(['progress' => 0, 'status' => 'not-enrolled']);
        }
        
        // Calculate actual progress
        $course = \App\Models\Course::with(['lessons.topics'])->find($courseId);
        $totalTopics = 0;
        $completedTopics = 0;
        
        foreach ($course->lessons as $lesson) {
            foreach ($lesson->topics as $topic) {
                if ($topic->is_published) {
                    $totalTopics++;
                    $progress = \App\Models\UserProgress::where('user_id', $user->id)
                        ->where('topic_id', $topic->id)
                        ->where('completed', true)
                        ->first();
                    
                    if ($progress) {
                        $completedTopics++;
                    }
                }
            }
        }
        
        $progressPercentage = $totalTopics > 0 ? round(($completedTopics / $totalTopics) * 100) : 0;
        
        return response()->json([
            'progress' => $progressPercentage,
            'completed_topics' => $completedTopics,
            'total_topics' => $totalTopics,
            'status' => $enrollment->status
        ]);
    });

    // Settings routes
    Route::get('/settings', [SettingsController::class, 'index']);
    Route::put('/settings', [SettingsController::class, 'update']);
    
    // User profile/preferences routes
    Route::put('/user/preferences', function (Request $request) {
        try {
            $user = auth()->user();
            $validated = $request->validate([
                'language' => 'nullable|string|max:10',
                'timezone' => 'nullable|string|max:50',
            ]);
            
            $user->update($validated);
            
            return response()->json([
                'success' => true,
                'message' => 'Preferences updated successfully',
                'user' => $user
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating preferences: ' . $e->getMessage()
            ], 500);
        }
    });

    // Study Time routes
    Route::get('/study-time/{courseId}/status', [StudyTimeController::class, 'getStatus']);
    Route::post('/study-time/{courseId}/start', [StudyTimeController::class, 'startSession']);
    Route::post('/study-time/{courseId}/end', [StudyTimeController::class, 'endSession']);
    Route::post('/study-time/{courseId}/add', [StudyTimeController::class, 'addTime']);
    Route::get('/study-time/{courseId}/stats', [StudyTimeController::class, 'getStats']);
    Route::get('/study-time/{courseId}/minimum', [StudyTimeController::class, 'checkMinimum']);

    // Survey routes
    Route::get('/courses/{courseId}/survey', [SurveyController::class, 'getCourseSurvey']);
    Route::post('/courses/{courseId}/survey/submit', [SurveyController::class, 'submitSurvey']);
    Route::get('/courses/{courseId}/survey/stats', [SurveyController::class, 'getStats']);
    Route::post('/courses/{courseId}/survey/questions', [SurveyController::class, 'manageQuestions']);

    // Legacy routes for backward compatibility (sections -> lessons)
    Route::get('/sections', [LessonController::class, 'index']); // ?course_id=123
    Route::post('/sections', [LessonController::class, 'store']);
    Route::get('/sections/{id}', [LessonController::class, 'show']);
    Route::put('/sections/{id}', [LessonController::class, 'update']);
    Route::delete('/sections/{id}', [LessonController::class, 'destroy']);

    Route::get('/get_sections', [LessonController::class, 'index']); // ?course_id=123
    Route::get('/get_section/{id}', [LessonController::class, 'show']);
    Route::post('/save_section', [LessonController::class, 'store']);
    Route::delete('/delete_section/{id}', [LessonController::class, 'destroy']);

    // Legacy topic routes
    Route::get('/get_topics', [TopicController::class, 'index']); // ?section_id=123
    Route::get('/get_topic/{id}', [TopicController::class, 'show']);
    Route::post('/save_topic', [TopicController::class, 'store']);
    Route::delete('/delete_topic/{id}', [TopicController::class, 'destroy']);

    // Legacy quiz routes
    Route::get('/get_quiz', [QuizController::class, 'getQuiz']); // ?topic_id=123
    Route::post('/save_quiz', [QuizController::class, 'store']);
    Route::delete('/delete_quiz/{id}', [QuizController::class, 'destroy']);

    // Legacy progress routes
    Route::post('/update_progress', [ProgressController::class, 'updateProgress']);
    Route::get('/get_resume_point', [ProgressController::class, 'getResumePoint']); // ?course_id=123
    Route::get('/user_progress', [ProgressController::class, 'getUserProgress']);

    // Legacy settings routes
    Route::get('/get_settings', [SettingsController::class, 'index']);

    // Legacy course routes
    Route::get('/get_courses', [CourseController::class, 'index']);
    Route::get('/get_course/{id}', [CourseController::class, 'show']);
    Route::post('/save_course', [CourseController::class, 'store']);
    Route::delete('/delete_course/{id}', [CourseController::class, 'destroy']);
});

// Public settings routes
Route::get('/settings/access-code-status', [ApiSettingsController::class, 'getAccessCodeStatus']);
Route::post('/settings/validate-access-code', [ApiSettingsController::class, 'validateAccessCode']);

// Admin settings routes
Route::middleware(['auth'])->group(function () {
    Route::post('/settings/access-code-toggle', [ApiSettingsController::class, 'toggleAccessCode']);
    Route::post('/settings/generate-access-codes', [ApiSettingsController::class, 'generateAccessCodes']);
    Route::get('/settings/access-codes', [ApiSettingsController::class, 'listAccessCodes']);
});